/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.AddressMessage;
import com.google.bitcoin.core.AlertMessage;
import com.google.bitcoin.core.Block;
import com.google.bitcoin.core.GetAddrMessage;
import com.google.bitcoin.core.GetBlocksMessage;
import com.google.bitcoin.core.GetDataMessage;
import com.google.bitcoin.core.GetHeadersMessage;
import com.google.bitcoin.core.HeadersMessage;
import com.google.bitcoin.core.InventoryMessage;
import com.google.bitcoin.core.Message;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.Ping;
import com.google.bitcoin.core.ProtocolException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.UnknownMessage;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VersionAck;
import com.google.bitcoin.core.VersionMessage;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class BitcoinSerializer {
    private static final Logger log = LoggerFactory.getLogger(BitcoinSerializer.class);
    private static final int COMMAND_LEN = 12;
    private NetworkParameters params;
    private boolean usesChecksumming;
    private boolean parseLazy = false;
    private boolean parseRetain = false;
    private static Map<Class<? extends Message>, String> names = new HashMap<Class<? extends Message>, String>();
    private LinkedHashMap<Sha256Hash, Integer> dedupeList;

    public static LinkedHashMap<Sha256Hash, Integer> createDedupeList() {
        return new LinkedHashMap<Sha256Hash, Integer>(){

            @Override
            protected boolean removeEldestEntry(Map.Entry<Sha256Hash, Integer> entry) {
                return this.size() > 100;
            }
        };
    }

    public BitcoinSerializer(NetworkParameters params, boolean usesChecksumming, LinkedHashMap<Sha256Hash, Integer> dedupeList) {
        this(params, usesChecksumming, false, false, dedupeList);
    }

    public BitcoinSerializer(NetworkParameters params, boolean usesChecksumming, boolean parseLazy, boolean parseRetain, LinkedHashMap<Sha256Hash, Integer> dedupeList) {
        this.params = params;
        this.usesChecksumming = usesChecksumming;
        this.dedupeList = dedupeList;
        this.parseLazy = parseLazy;
        this.parseRetain = parseRetain;
    }

    public void setUseChecksumming(boolean usesChecksumming) {
        this.usesChecksumming = usesChecksumming;
    }

    public boolean getUseChecksumming() {
        return this.usesChecksumming;
    }

    public int getHeaderLength() {
        return 20 + (this.usesChecksumming ? 4 : 0);
    }

    public void serialize(Message message, OutputStream out) throws IOException {
        String name = names.get(message.getClass());
        if (name == null) {
            throw new Error("BitcoinSerializer doesn't currently know how to serialize " + message.getClass());
        }
        byte[] header = new byte[20 + (this.usesChecksumming ? 4 : 0)];
        Utils.uint32ToByteArrayBE(this.params.packetMagic, header, 0);
        for (int i = 0; i < name.length() && i < 12; ++i) {
            header[4 + i] = (byte)(name.codePointAt(i) & 0xFF);
        }
        byte[] payload = message.bitcoinSerialize();
        Utils.uint32ToByteArrayLE(payload.length, header, 16);
        if (this.usesChecksumming) {
            byte[] checksum = message.getChecksum();
            if (checksum == null) {
                Sha256Hash msgHash = message.getHash();
                if (msgHash != null && message instanceof Transaction) {
                    int start;
                    byte[] hash = msgHash.getBytes();
                    for (int i = start = 20; i < start + 4; ++i) {
                        header[i] = hash[31 - i + start];
                    }
                } else {
                    byte[] hash = Utils.doubleDigest(payload);
                    System.arraycopy(hash, 0, header, 20, 4);
                }
            } else {
                assert (Arrays.equals(checksum, Utils.copyOf(Utils.doubleDigest(payload), 4))) : "Checksum match failure on serialization.  Cached: " + Arrays.toString(checksum) + " Calculated: " + Arrays.toString(Utils.copyOf(Utils.doubleDigest(payload), 4));
                System.arraycopy(checksum, 0, header, 20, 4);
            }
        }
        out.write(header);
        out.write(payload);
        if (log.isDebugEnabled()) {
            log.debug("Sending {} message: {}", (Object)name, (Object)(Utils.bytesToHexString(header) + Utils.bytesToHexString(payload)));
        }
    }

    public Message deserialize(InputStream in) throws ProtocolException, IOException {
        this.seekPastMagicBytes(in);
        BitcoinPacketHeader header = new BitcoinPacketHeader(this.usesChecksumming, in);
        return this.deserializePayload(header, in);
    }

    private boolean canDedupeMessageType(String command) {
        return command.equals("block") || command.equals("tx");
    }

    public BitcoinPacketHeader deserializeHeader(InputStream in) throws ProtocolException, IOException {
        return new BitcoinPacketHeader(this.usesChecksumming, in);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Message deserializePayload(BitcoinPacketHeader header, InputStream in) throws ProtocolException, IOException {
        int bytesRead;
        byte[] payloadBytes = new byte[header.size];
        for (int readCursor = 0; readCursor < payloadBytes.length - 1; readCursor += bytesRead) {
            bytesRead = in.read(payloadBytes, readCursor, header.size - readCursor);
            if (bytesRead != -1) continue;
            throw new IOException("Socket is disconnected");
        }
        Sha256Hash singleHash = null;
        if (this.dedupeList != null && this.canDedupeMessageType(header.command)) {
            LinkedHashMap<Sha256Hash, Integer> linkedHashMap = this.dedupeList;
            synchronized (linkedHashMap) {
                singleHash = Sha256Hash.create(payloadBytes);
                Integer count = this.dedupeList.get(singleHash);
                if (count != null) {
                    int newCount = count + 1;
                    log.info("Received duplicate {} message, now seen {} times", (Object)header.command, (Object)newCount);
                    this.dedupeList.put(singleHash, newCount);
                    return null;
                }
                this.dedupeList.put(singleHash, 1);
            }
        }
        byte[] hash = null;
        if (this.usesChecksumming && (header.checksum[0] != (hash = singleHash != null ? Utils.singleDigest(singleHash.getBytes(), 0, 32) : Utils.doubleDigest(payloadBytes))[0] || header.checksum[1] != hash[1] || header.checksum[2] != hash[2] || header.checksum[3] != hash[3])) {
            throw new ProtocolException("Checksum failed to verify, actual " + Utils.bytesToHexString(hash) + " vs " + Utils.bytesToHexString(header.checksum));
        }
        if (log.isDebugEnabled()) {
            log.debug("Received {} byte '{}' message: {}", new Object[]{header.size, header.command, Utils.bytesToHexString(payloadBytes)});
        }
        try {
            return this.makeMessage(header.command, header.size, payloadBytes, hash, header.checksum);
        }
        catch (Exception e) {
            throw new ProtocolException("Error deserializing message " + Utils.bytesToHexString(payloadBytes) + "\n", e);
        }
    }

    private Message makeMessage(String command, int length, byte[] payloadBytes, byte[] hash, byte[] checksum) throws ProtocolException {
        Message message;
        if (command.equals("version")) {
            return new VersionMessage(this.params, payloadBytes);
        }
        if (command.equals("inv")) {
            message = new InventoryMessage(this.params, payloadBytes, this.parseLazy, this.parseRetain, length);
        } else if (command.equals("block")) {
            message = new Block(this.params, payloadBytes, this.parseLazy, this.parseRetain, length);
        } else if (command.equals("getdata")) {
            message = new GetDataMessage(this.params, payloadBytes, this.parseLazy, this.parseRetain, length);
        } else if (command.equals("tx")) {
            Transaction tx = new Transaction(this.params, payloadBytes, null, this.parseLazy, this.parseRetain, length);
            if (hash != null) {
                tx.setHash(new Sha256Hash(Utils.reverseBytes(hash)));
            }
            message = tx;
        } else if (command.equals("addr")) {
            message = new AddressMessage(this.params, payloadBytes, this.parseLazy, this.parseRetain, length);
        } else {
            if (command.equals("ping")) {
                return new Ping();
            }
            if (command.equals("verack")) {
                return new VersionAck(this.params, payloadBytes);
            }
            if (command.equals("headers")) {
                return new HeadersMessage(this.params, payloadBytes);
            }
            if (command.equals("alert")) {
                return new AlertMessage(this.params, payloadBytes);
            }
            log.warn("No support for deserializing message with name {}", (Object)command);
            return new UnknownMessage(this.params, command, payloadBytes);
        }
        if (checksum != null) {
            message.setChecksum(checksum);
        }
        return message;
    }

    public void seekPastMagicBytes(InputStream in) throws IOException {
        int magicCursor = 3;
        while (true) {
            int b;
            if ((b = in.read()) == -1) {
                throw new IOException("Socket is disconnected");
            }
            int expectedByte = 0xFF & (int)(this.params.packetMagic >>> magicCursor * 8);
            if (b == expectedByte) {
                if (--magicCursor >= 0) continue;
                return;
            }
            magicCursor = 3;
        }
    }

    public boolean isParseLazyMode() {
        return this.parseLazy;
    }

    public boolean isParseRetainMode() {
        return this.parseRetain;
    }

    static {
        names.put(VersionMessage.class, "version");
        names.put(InventoryMessage.class, "inv");
        names.put(Block.class, "block");
        names.put(GetDataMessage.class, "getdata");
        names.put(Transaction.class, "tx");
        names.put(AddressMessage.class, "addr");
        names.put(Ping.class, "ping");
        names.put(VersionAck.class, "verack");
        names.put(GetBlocksMessage.class, "getblocks");
        names.put(GetHeadersMessage.class, "getheaders");
        names.put(GetAddrMessage.class, "getaddr");
        names.put(HeadersMessage.class, "headers");
    }

    public class BitcoinPacketHeader {
        final byte[] header;
        final String command;
        final int size;
        final byte[] checksum;

        BitcoinPacketHeader(boolean usesCheckSumminng, InputStream in) throws ProtocolException, IOException {
            int cursor;
            int bytesRead;
            this.header = new byte[16 + (BitcoinSerializer.this.usesChecksumming ? 4 : 0)];
            for (int readCursor = 0; readCursor < this.header.length; readCursor += bytesRead) {
                bytesRead = in.read(this.header, readCursor, this.header.length - readCursor);
                if (bytesRead != -1) continue;
                throw new IOException("Incomplete packet in underlying stream");
            }
            int mark = cursor = 0;
            while (this.header[cursor] != 0 && cursor - mark < 12) {
                ++cursor;
            }
            byte[] commandBytes = new byte[cursor - mark];
            System.arraycopy(this.header, mark, commandBytes, 0, cursor - mark);
            try {
                this.command = new String(commandBytes, "US-ASCII");
            }
            catch (UnsupportedEncodingException e) {
                throw new RuntimeException(e);
            }
            cursor = mark + 12;
            this.size = (int)Utils.readUint32(this.header, cursor);
            cursor += 4;
            if (this.size > 0x2000000) {
                throw new ProtocolException("Message size too large: " + this.size);
            }
            this.checksum = new byte[4];
            if (BitcoinSerializer.this.usesChecksumming) {
                System.arraycopy(this.header, cursor, this.checksum, 0, 4);
                cursor += 4;
            }
        }

        public boolean hasCheckSum() {
            return this.checksum != null;
        }

        public byte[] getHeader() {
            return this.header;
        }

        public String getCommand() {
            return this.command;
        }

        public int getPayloadSize() {
            return this.size;
        }

        public byte[] getChecksum() {
            return this.checksum;
        }
    }
}

