/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.Address;
import com.google.bitcoin.core.Message;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.ProtocolException;
import com.google.bitcoin.core.Script;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.TransactionInput;
import com.google.bitcoin.core.TransactionOutput;
import com.google.bitcoin.core.UnsafeByteArrayOutputStream;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VarInt;
import com.google.bitcoin.core.VerificationException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class Block
extends Message {
    private static final Logger log = LoggerFactory.getLogger(Block.class);
    private static final long serialVersionUID = 2738848929966035281L;
    public static final int HEADER_SIZE = 80;
    static final long ALLOWED_TIME_DRIFT = 7200L;
    static final long EASIEST_DIFFICULTY_TARGET = 545259519L;
    static long fakeClock = 0L;
    private long version;
    private Sha256Hash prevBlockHash;
    private Sha256Hash merkleRoot;
    private long time;
    private long difficultyTarget;
    private long nonce;
    List<Transaction> transactions;
    private transient Sha256Hash hash;
    private transient boolean headerParsed;
    private transient boolean transactionsParsed;
    private transient boolean headerBytesValid;
    private transient boolean transactionBytesValid;
    private static BigInteger LARGEST_HASH = BigInteger.ONE.shiftLeft(256);
    private static int txCounter;
    static final byte[] EMPTY_BYTES;

    Block(NetworkParameters params) {
        super(params);
        this.version = 1L;
        this.difficultyTarget = 487063544L;
        this.time = System.currentTimeMillis() / 1000L;
        this.prevBlockHash = Sha256Hash.ZERO_HASH;
        this.length = 80;
    }

    public Block(NetworkParameters params, byte[] payloadBytes) throws ProtocolException {
        super(params, payloadBytes, 0);
    }

    public Block(NetworkParameters params, byte[] payloadBytes, boolean parseLazy, boolean parseRetain, int length) throws ProtocolException {
        super(params, payloadBytes, 0, parseLazy, parseRetain, length);
    }

    private void readObject(ObjectInputStream ois) throws ClassNotFoundException, IOException {
        ois.defaultReadObject();
        this.hash = null;
    }

    private void parseHeader() {
        if (this.headerParsed) {
            return;
        }
        this.cursor = this.offset;
        this.version = this.readUint32();
        this.prevBlockHash = this.readHash();
        this.merkleRoot = this.readHash();
        this.time = this.readUint32();
        this.difficultyTarget = this.readUint32();
        this.nonce = this.readUint32();
        this.hash = new Sha256Hash(Utils.reverseBytes(Utils.doubleDigest(this.bytes, this.offset, this.cursor)));
        this.headerParsed = true;
        this.headerBytesValid = this.parseRetain;
    }

    private void parseTransactions() throws ProtocolException {
        if (this.transactionsParsed) {
            return;
        }
        this.cursor = this.offset + 80;
        if (this.bytes.length == this.cursor) {
            this.transactionsParsed = true;
            this.transactionBytesValid = false;
            return;
        }
        int numTransactions = (int)this.readVarInt();
        this.transactions = new ArrayList<Transaction>(numTransactions);
        for (int i = 0; i < numTransactions; ++i) {
            Transaction tx = new Transaction(this.params, this.bytes, this.cursor, this, this.parseLazy, this.parseRetain, -1);
            this.transactions.add(tx);
            this.cursor += tx.getMessageSize();
        }
        this.transactionsParsed = true;
        this.transactionBytesValid = this.parseRetain;
    }

    @Override
    void parse() throws ProtocolException {
        this.parseHeader();
        this.parseTransactions();
        this.length = this.cursor - this.offset;
    }

    @Override
    protected void parseLite() throws ProtocolException {
        if (this.length == -1) {
            assert (!this.parseLazy) : "Performing lite parse of block transaction as block was initialised from byte array without providing length.  This should never need to happen. parseLazy: " + this.parseLazy;
            this.parseTransactions();
            this.length = this.cursor - this.offset;
        } else {
            this.transactionBytesValid = !this.transactionsParsed || this.parseRetain && this.length > 80;
        }
        this.headerBytesValid = !this.headerParsed || this.parseRetain && this.length >= 80;
    }

    private synchronized void maybeParseHeader() {
        if (this.headerParsed || this.bytes == null) {
            return;
        }
        this.parseHeader();
        if (!this.headerBytesValid && !this.transactionBytesValid) {
            this.bytes = null;
        }
    }

    private synchronized void maybeParseTransactions() {
        if (this.transactionsParsed || this.bytes == null) {
            return;
        }
        try {
            this.parseTransactions();
            if (!this.parseRetain) {
                this.transactionBytesValid = false;
                if (this.headerParsed) {
                    this.bytes = null;
                }
            }
        }
        catch (ProtocolException e) {
            throw new Message.LazyParseException("ProtocolException caught during lazy parse.  For safe access to fields call ensureParsed before attempting read or write access", e);
        }
    }

    @Override
    protected synchronized void maybeParse() {
        throw new Message.LazyParseException("checkParse() should never be called on a Block.  Instead use checkParseHeader() and checkParseTransactions()");
    }

    @Override
    public void ensureParsed() throws ProtocolException {
        try {
            this.maybeParseHeader();
            this.maybeParseTransactions();
        }
        catch (Message.LazyParseException e) {
            if (e.getCause() instanceof ProtocolException) {
                throw (ProtocolException)e.getCause();
            }
            throw new ProtocolException(e);
        }
    }

    public void ensureParsedHeader() throws ProtocolException {
        try {
            this.maybeParseHeader();
        }
        catch (Message.LazyParseException e) {
            if (e.getCause() instanceof ProtocolException) {
                throw (ProtocolException)e.getCause();
            }
            throw new ProtocolException(e);
        }
    }

    public void ensureParsedTransactions() throws ProtocolException {
        try {
            this.maybeParseTransactions();
        }
        catch (Message.LazyParseException e) {
            if (e.getCause() instanceof ProtocolException) {
                throw (ProtocolException)e.getCause();
            }
            throw new ProtocolException(e);
        }
    }

    private void writeHeader(OutputStream stream) throws IOException {
        if (this.headerBytesValid && this.bytes != null && this.bytes.length >= this.offset + 80) {
            stream.write(this.bytes, this.offset, 80);
            return;
        }
        this.maybeParseHeader();
        Utils.uint32ToByteStreamLE(this.version, stream);
        stream.write(Utils.reverseBytes(this.prevBlockHash.getBytes()));
        stream.write(Utils.reverseBytes(this.getMerkleRoot().getBytes()));
        Utils.uint32ToByteStreamLE(this.time, stream);
        Utils.uint32ToByteStreamLE(this.difficultyTarget, stream);
        Utils.uint32ToByteStreamLE(this.nonce, stream);
    }

    private void writeTransactions(OutputStream stream) throws IOException {
        if (this.transactions == null && this.transactionsParsed) {
            return;
        }
        if (this.transactionBytesValid && this.bytes != null && this.bytes.length >= this.offset + this.length) {
            stream.write(this.bytes, this.offset + 80, this.length - 80);
            return;
        }
        if (this.transactions != null) {
            stream.write(new VarInt(this.transactions.size()).encode());
            for (Transaction tx : this.transactions) {
                tx.bitcoinSerialize(stream);
            }
        }
    }

    @Override
    public byte[] bitcoinSerialize() {
        if (this.headerBytesValid && this.transactionBytesValid) {
            assert (this.bytes != null) : "Bytes should never be null if headerBytesValid && transactionBytesValid";
            if (this.length == this.bytes.length) {
                return this.bytes;
            }
            byte[] buf = new byte[this.length];
            System.arraycopy(this.bytes, this.offset, buf, 0, this.length);
            return buf;
        }
        UnsafeByteArrayOutputStream stream = new UnsafeByteArrayOutputStream(this.length == -1 ? 80 + this.guessTransactionsLength() : this.length);
        try {
            this.writeHeader(stream);
            this.writeTransactions(stream);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return ((ByteArrayOutputStream)stream).toByteArray();
    }

    @Override
    protected void bitcoinSerializeToStream(OutputStream stream) throws IOException {
        this.writeHeader(stream);
        this.writeTransactions(stream);
    }

    private int guessTransactionsLength() {
        if (this.transactionBytesValid) {
            return this.bytes.length - 80;
        }
        if (this.transactions == null) {
            return 0;
        }
        int len = VarInt.sizeOf(this.transactions.size());
        for (Transaction tx : this.transactions) {
            len += tx.length == -1 ? 255 : tx.length;
        }
        return len;
    }

    @Override
    protected void unCache() {
        this.unCacheTransactions();
    }

    private void unCacheHeader() {
        this.maybeParseHeader();
        this.headerBytesValid = false;
        if (!this.transactionBytesValid) {
            this.bytes = null;
        }
        this.hash = null;
        this.checksum = null;
    }

    private void unCacheTransactions() {
        this.maybeParseTransactions();
        this.transactionBytesValid = false;
        if (!this.headerBytesValid) {
            this.bytes = null;
        }
        this.unCacheHeader();
        this.merkleRoot = null;
    }

    private Sha256Hash calculateHash() {
        try {
            UnsafeByteArrayOutputStream bos = new UnsafeByteArrayOutputStream(80);
            this.writeHeader(bos);
            return new Sha256Hash(Utils.reverseBytes(Utils.doubleDigest(((ByteArrayOutputStream)bos).toByteArray())));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public String getHashAsString() {
        return this.getHash().toString();
    }

    @Override
    public Sha256Hash getHash() {
        if (this.hash == null) {
            this.hash = this.calculateHash();
        }
        return this.hash;
    }

    public BigInteger getWork() throws VerificationException {
        BigInteger target = this.getDifficultyTargetAsInteger();
        return LARGEST_HASH.divide(target.add(BigInteger.ONE));
    }

    public Block cloneAsHeader() {
        this.maybeParseHeader();
        Block block = new Block(this.params);
        block.nonce = this.nonce;
        block.prevBlockHash = this.prevBlockHash.duplicate();
        block.merkleRoot = this.getMerkleRoot().duplicate();
        block.version = this.version;
        block.time = this.time;
        block.difficultyTarget = this.difficultyTarget;
        block.transactions = null;
        block.hash = this.getHash().duplicate();
        return block;
    }

    public String toString() {
        StringBuffer s = new StringBuffer("v" + this.version + " block: \n" + "   previous block: " + this.prevBlockHash.toString() + "\n" + "   merkle root: " + this.getMerkleRoot().toString() + "\n" + "   time: [" + this.time + "] " + new Date(this.time * 1000L).toString() + "\n" + "   difficulty target (nBits): " + this.difficultyTarget + "\n" + "   nonce: " + this.nonce + "\n");
        if (this.transactions != null && this.transactions.size() > 0) {
            s.append("   with ").append(this.transactions.size()).append(" transaction(s):\n");
            for (Transaction tx : this.transactions) {
                s.append(tx.toString());
            }
        }
        return s.toString();
    }

    void solve() {
        this.maybeParseHeader();
        try {
            while (true) {
                if (this.checkProofOfWork(false)) {
                    return;
                }
                this.setNonce(this.getNonce() + 1L);
            }
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
    }

    public BigInteger getDifficultyTargetAsInteger() throws VerificationException {
        this.maybeParseHeader();
        BigInteger target = Utils.decodeCompactBits(this.difficultyTarget);
        if (target.compareTo(BigInteger.valueOf(0L)) <= 0 || target.compareTo(this.params.proofOfWorkLimit) > 0) {
            throw new VerificationException("Difficulty target is bad: " + target.toString());
        }
        return target;
    }

    private boolean checkProofOfWork(boolean throwException) throws VerificationException {
        BigInteger target = this.getDifficultyTargetAsInteger();
        BigInteger h = this.getHash().toBigInteger();
        if (h.compareTo(target) > 0) {
            if (throwException) {
                throw new VerificationException("Hash is higher than target: " + this.getHashAsString() + " vs " + target.toString(16));
            }
            return false;
        }
        return true;
    }

    private void checkTimestamp() throws VerificationException {
        long currentTime;
        this.maybeParseHeader();
        long l = currentTime = fakeClock != 0L ? fakeClock : System.currentTimeMillis() / 1000L;
        if (this.time > currentTime + 7200L) {
            throw new VerificationException("Block too far in future");
        }
    }

    private void checkMerkleRoot() throws VerificationException {
        Sha256Hash calculatedRoot = this.calculateMerkleRoot();
        if (!calculatedRoot.equals(this.merkleRoot)) {
            log.error("Merkle tree did not verify");
            throw new VerificationException("Merkle hashes do not match: " + calculatedRoot + " vs " + this.merkleRoot);
        }
    }

    private Sha256Hash calculateMerkleRoot() {
        List<byte[]> tree = this.buildMerkleTree();
        return new Sha256Hash(tree.get(tree.size() - 1));
    }

    private List<byte[]> buildMerkleTree() {
        this.maybeParseTransactions();
        ArrayList<byte[]> tree = new ArrayList<byte[]>();
        for (Transaction t : this.transactions) {
            tree.add(t.getHash().getBytes());
        }
        int levelOffset = 0;
        int levelSize = this.transactions.size();
        while (levelSize > 1) {
            for (int left = 0; left < levelSize; left += 2) {
                int right = Math.min(left + 1, levelSize - 1);
                byte[] leftBytes = Utils.reverseBytes(tree.get(levelOffset + left));
                byte[] rightBytes = Utils.reverseBytes(tree.get(levelOffset + right));
                tree.add(Utils.reverseBytes(Utils.doubleDigestTwoBuffers(leftBytes, 0, 32, rightBytes, 0, 32)));
            }
            levelOffset += levelSize;
            levelSize = (levelSize + 1) / 2;
        }
        return tree;
    }

    private void checkTransactions() throws VerificationException {
        if (!this.transactions.get(0).isCoinBase()) {
            throw new VerificationException("First tx is not coinbase");
        }
        for (int i = 1; i < this.transactions.size(); ++i) {
            if (!this.transactions.get(i).isCoinBase()) continue;
            throw new VerificationException("TX " + i + " is coinbase when it should not be.");
        }
    }

    public void verifyHeader() throws VerificationException {
        this.maybeParseHeader();
        this.checkProofOfWork(true);
        this.checkTimestamp();
    }

    public void verifyTransactions() throws VerificationException {
        assert (this.transactions.size() > 0);
        this.maybeParseTransactions();
        this.checkTransactions();
        this.checkMerkleRoot();
    }

    public void verify() throws VerificationException {
        this.verifyHeader();
        this.verifyTransactions();
    }

    public boolean equals(Object o) {
        if (!(o instanceof Block)) {
            return false;
        }
        Block other = (Block)o;
        return this.getHash().equals(other.getHash());
    }

    public int hashCode() {
        return this.getHash().hashCode();
    }

    public Sha256Hash getMerkleRoot() {
        this.maybeParseHeader();
        if (this.merkleRoot == null) {
            this.unCacheHeader();
            this.merkleRoot = this.calculateMerkleRoot();
        }
        return this.merkleRoot;
    }

    void setMerkleRoot(Sha256Hash value) {
        this.unCacheHeader();
        this.merkleRoot = value;
        this.hash = null;
    }

    void addTransaction(Transaction t) {
        this.unCacheTransactions();
        if (this.transactions == null) {
            this.transactions = new ArrayList<Transaction>();
        }
        t.setParent(this);
        this.transactions.add(t);
        this.adjustLength(t.length);
        this.merkleRoot = null;
        this.hash = null;
    }

    public long getVersion() {
        this.maybeParseHeader();
        return this.version;
    }

    public Sha256Hash getPrevBlockHash() {
        this.maybeParseHeader();
        return this.prevBlockHash;
    }

    void setPrevBlockHash(Sha256Hash prevBlockHash) {
        this.unCacheHeader();
        this.prevBlockHash = prevBlockHash;
        this.hash = null;
    }

    public long getTimeSeconds() {
        this.maybeParseHeader();
        return this.time;
    }

    public Date getTime() {
        return new Date(this.getTimeSeconds() * 1000L);
    }

    void setTime(long time) {
        this.unCacheHeader();
        this.time = time;
        this.hash = null;
    }

    public long getDifficultyTarget() {
        this.maybeParseHeader();
        return this.difficultyTarget;
    }

    void setDifficultyTarget(long compactForm) {
        this.unCacheHeader();
        this.difficultyTarget = compactForm;
        this.hash = null;
    }

    public long getNonce() {
        this.maybeParseHeader();
        return this.nonce;
    }

    void setNonce(long nonce) {
        this.unCacheHeader();
        this.nonce = nonce;
        this.hash = null;
    }

    public List<Transaction> getTransactions() {
        this.maybeParseTransactions();
        return Collections.unmodifiableList(this.transactions);
    }

    void addCoinbaseTransaction(byte[] pubKeyTo) {
        this.unCacheTransactions();
        this.transactions = new ArrayList<Transaction>();
        Transaction coinbase = new Transaction(this.params);
        coinbase.addInput(new TransactionInput(this.params, coinbase, new byte[]{(byte)txCounter++}));
        coinbase.addOutput(new TransactionOutput(this.params, coinbase, Script.createOutputScript(pubKeyTo)));
        this.transactions.add(coinbase);
    }

    Block createNextBlock(Address to, long time) {
        Block b = new Block(this.params);
        b.setDifficultyTarget(this.difficultyTarget);
        b.addCoinbaseTransaction(EMPTY_BYTES);
        Transaction t = new Transaction(this.params);
        t.addOutput(new TransactionOutput(this.params, t, Utils.toNanoCoins(50, 0), to));
        TransactionInput input = new TransactionInput(this.params, t, Script.createInputScript(EMPTY_BYTES, EMPTY_BYTES));
        byte[] counter = new byte[32];
        counter[0] = (byte)txCounter++;
        input.getOutpoint().setHash(new Sha256Hash(counter));
        t.addInput(input);
        b.addTransaction(t);
        b.setPrevBlockHash(this.getHash());
        b.setTime(time);
        b.solve();
        try {
            b.verifyHeader();
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
        return b;
    }

    public Block createNextBlock(Address to) {
        return this.createNextBlock(to, Utils.now().getTime() / 1000L);
    }

    boolean isParsedHeader() {
        return this.headerParsed;
    }

    boolean isParsedTransactions() {
        return this.transactionsParsed;
    }

    boolean isHeaderBytesValid() {
        return this.headerBytesValid;
    }

    boolean isTransactionBytesValid() {
        return this.transactionBytesValid;
    }

    static {
        EMPTY_BYTES = new byte[32];
    }
}

