/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.Block;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.ScriptException;
import com.google.bitcoin.core.StoredBlock;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VerificationException;
import com.google.bitcoin.core.Wallet;
import com.google.bitcoin.store.BlockStore;
import com.google.bitcoin.store.BlockStoreException;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class BlockChain {
    private static final Logger log = LoggerFactory.getLogger(BlockChain.class);
    protected final BlockStore blockStore;
    protected StoredBlock chainHead;
    protected final Object chainHeadLock = new Object();
    protected final NetworkParameters params;
    protected final List<Wallet> wallets;
    private final ArrayList<Block> unconnectedBlocks = new ArrayList();
    private long statsLastTime = System.currentTimeMillis();
    private long statsBlocksAdded;

    public BlockChain(NetworkParameters params, Wallet wallet, BlockStore blockStore) throws BlockStoreException {
        this(params, new ArrayList<Wallet>(), blockStore);
        if (wallet != null) {
            this.addWallet(wallet);
        }
    }

    public BlockChain(NetworkParameters params, BlockStore blockStore) throws BlockStoreException {
        this(params, new ArrayList<Wallet>(), blockStore);
    }

    public BlockChain(NetworkParameters params, List<Wallet> wallets, BlockStore blockStore) throws BlockStoreException {
        this.blockStore = blockStore;
        this.chainHead = blockStore.getChainHead();
        log.info("chain head is at height {}:\n{}", (Object)this.chainHead.getHeight(), (Object)this.chainHead.getHeader());
        this.params = params;
        this.wallets = new ArrayList<Wallet>(wallets);
    }

    public synchronized void addWallet(Wallet wallet) {
        this.wallets.add(wallet);
    }

    public BlockStore getBlockStore() {
        return this.blockStore;
    }

    public synchronized boolean add(Block block) throws VerificationException, ScriptException {
        try {
            return this.add(block, true);
        }
        catch (BlockStoreException e) {
            throw new RuntimeException(e);
        }
    }

    private synchronized boolean add(Block block, boolean tryConnecting) throws BlockStoreException, VerificationException, ScriptException {
        if (System.currentTimeMillis() - this.statsLastTime > 1000L) {
            if (this.statsBlocksAdded > 1L) {
                log.info("{} blocks per second", (Object)this.statsBlocksAdded);
            }
            this.statsLastTime = System.currentTimeMillis();
            this.statsBlocksAdded = 0L;
        }
        if (block.equals(this.chainHead.getHeader())) {
            log.debug("Chain head added more than once: {}", (Object)block.getHash());
            return true;
        }
        boolean contentsImportant = false;
        HashMap<Wallet, List<Transaction>> walletToTxMap = new HashMap<Wallet, List<Transaction>>();
        if (block.transactions != null) {
            this.scanTransactions(block, walletToTxMap);
            contentsImportant = walletToTxMap.size() > 0;
        }
        try {
            block.verifyHeader();
            if (contentsImportant) {
                block.verifyTransactions();
            }
        }
        catch (VerificationException e) {
            log.error("Failed to verify block: ", (Throwable)e);
            log.error(block.getHashAsString());
            throw e;
        }
        StoredBlock storedPrev = this.blockStore.get(block.getPrevBlockHash());
        if (storedPrev == null) {
            assert (tryConnecting) : "bug in tryConnectingUnconnected";
            log.warn("Block does not connect: {}", (Object)block.getHashAsString());
            this.unconnectedBlocks.add(block);
            return false;
        }
        StoredBlock newStoredBlock = storedPrev.build(block);
        this.checkDifficultyTransitions(storedPrev, newStoredBlock);
        this.blockStore.put(newStoredBlock);
        this.connectBlock(newStoredBlock, storedPrev, walletToTxMap);
        if (tryConnecting) {
            this.tryConnectingUnconnected();
        }
        ++this.statsBlocksAdded;
        return true;
    }

    private void connectBlock(StoredBlock newStoredBlock, StoredBlock storedPrev, HashMap<Wallet, List<Transaction>> newTransactions) throws BlockStoreException, VerificationException {
        if (storedPrev.equals(this.chainHead)) {
            this.setChainHead(newStoredBlock);
            log.debug("Chain is now {} blocks high", (Object)this.chainHead.getHeight());
            if (newTransactions != null) {
                this.sendTransactionsToWallet(newStoredBlock, NewBlockType.BEST_CHAIN, newTransactions);
            }
        } else {
            boolean haveNewBestChain = newStoredBlock.moreWorkThan(this.chainHead);
            if (haveNewBestChain) {
                log.info("Block is causing a re-organize");
            } else {
                StoredBlock splitPoint = this.findSplit(newStoredBlock, this.chainHead);
                if (splitPoint == newStoredBlock) {
                    log.debug("Saw duplicated block in main chain at height {}: {}", (Object)newStoredBlock.getHeight(), (Object)newStoredBlock.getHeader().getHash());
                    return;
                }
                int splitPointHeight = splitPoint != null ? splitPoint.getHeight() : -1;
                String splitPointHash = splitPoint != null ? splitPoint.getHeader().getHashAsString() : "?";
                log.info("Block forks the chain at height {}/block {}, but it did not cause a reorganize:\n{}", new Object[]{splitPointHeight, splitPointHash, newStoredBlock});
            }
            if (newTransactions != null) {
                this.sendTransactionsToWallet(newStoredBlock, NewBlockType.SIDE_CHAIN, newTransactions);
            }
            if (haveNewBestChain) {
                this.handleNewBestChain(newStoredBlock);
            }
        }
    }

    private void handleNewBestChain(StoredBlock newChainHead) throws BlockStoreException, VerificationException {
        StoredBlock splitPoint = this.findSplit(newChainHead, this.chainHead);
        log.info("Re-organize after split at height {}", (Object)splitPoint.getHeight());
        log.info("Old chain head: {}", (Object)this.chainHead.getHeader().getHashAsString());
        log.info("New chain head: {}", (Object)newChainHead.getHeader().getHashAsString());
        log.info("Split at block: {}", (Object)splitPoint.getHeader().getHashAsString());
        List<StoredBlock> oldBlocks = this.getPartialChain(this.chainHead, splitPoint);
        List<StoredBlock> newBlocks = this.getPartialChain(newChainHead, splitPoint);
        for (Wallet wallet : this.wallets) {
            wallet.reorganize(oldBlocks, newBlocks);
        }
        this.setChainHead(newChainHead);
    }

    private List<StoredBlock> getPartialChain(StoredBlock higher, StoredBlock lower) throws BlockStoreException {
        assert (higher.getHeight() > lower.getHeight());
        LinkedList<StoredBlock> results = new LinkedList<StoredBlock>();
        StoredBlock cursor = higher;
        do {
            results.add(cursor);
            cursor = cursor.getPrev(this.blockStore);
            assert (cursor != null) : "Ran off the end of the chain";
        } while (!cursor.equals(lower));
        return results;
    }

    private StoredBlock findSplit(StoredBlock newChainHead, StoredBlock chainHead) throws BlockStoreException {
        StoredBlock currentChainCursor = chainHead;
        StoredBlock newChainCursor = newChainHead;
        while (!currentChainCursor.equals(newChainCursor)) {
            if (currentChainCursor.getHeight() > newChainCursor.getHeight()) {
                currentChainCursor = currentChainCursor.getPrev(this.blockStore);
                assert (currentChainCursor != null) : "Attempt to follow an orphan chain";
                continue;
            }
            newChainCursor = newChainCursor.getPrev(this.blockStore);
            assert (newChainCursor != null) : "Attempt to follow an orphan chain";
        }
        return currentChainCursor;
    }

    public int getBestChainHeight() {
        return this.getChainHead().getHeight();
    }

    private void sendTransactionsToWallet(StoredBlock block, NewBlockType blockType, HashMap<Wallet, List<Transaction>> newTransactions) throws VerificationException {
        for (Map.Entry<Wallet, List<Transaction>> entry : newTransactions.entrySet()) {
            try {
                List<Transaction> txns = entry.getValue();
                for (Transaction tx : txns) {
                    entry.getKey().receiveFromBlock(tx, block, blockType);
                }
            }
            catch (ScriptException e) {
                log.warn("Failed to parse a script: " + e.toString());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setChainHead(StoredBlock chainHead) throws BlockStoreException {
        this.blockStore.setChainHead(chainHead);
        Object object = this.chainHeadLock;
        synchronized (object) {
            this.chainHead = chainHead;
        }
    }

    private void tryConnectingUnconnected() throws VerificationException, ScriptException, BlockStoreException {
        int blocksConnectedThisRound;
        do {
            blocksConnectedThisRound = 0;
            Iterator<Block> iter = this.unconnectedBlocks.iterator();
            while (iter.hasNext()) {
                Block block = iter.next();
                log.debug("Trying to connect {}", (Object)block.getHash());
                StoredBlock prev = this.blockStore.get(block.getPrevBlockHash());
                if (prev == null) {
                    log.debug("  but it is not connectable right now");
                    continue;
                }
                this.add(block, false);
                iter.remove();
                ++blocksConnectedThisRound;
            }
            if (blocksConnectedThisRound <= 0) continue;
            log.info("Connected {} floating blocks.", (Object)blocksConnectedThisRound);
        } while (blocksConnectedThisRound > 0);
    }

    private void checkDifficultyTransitions(StoredBlock storedPrev, StoredBlock storedNext) throws BlockStoreException, VerificationException {
        Block prev = storedPrev.getHeader();
        Block next = storedNext.getHeader();
        if ((storedPrev.getHeight() + 1) % this.params.interval != 0) {
            if (next.getDifficultyTarget() != prev.getDifficultyTarget()) {
                throw new VerificationException("Unexpected change in difficulty at height " + storedPrev.getHeight() + ": " + Long.toHexString(next.getDifficultyTarget()) + " vs " + Long.toHexString(prev.getDifficultyTarget()));
            }
            return;
        }
        long now = System.currentTimeMillis();
        StoredBlock cursor = this.blockStore.get(prev.getHash());
        for (int i = 0; i < this.params.interval - 1; ++i) {
            if (cursor == null) {
                throw new VerificationException("Difficulty transition point but we did not find a way back to the genesis block.");
            }
            cursor = this.blockStore.get(cursor.getHeader().getPrevBlockHash());
        }
        log.debug("Difficulty transition traversal took {}msec", (Object)(System.currentTimeMillis() - now));
        Block blockIntervalAgo = cursor.getHeader();
        int timespan = (int)(prev.getTimeSeconds() - blockIntervalAgo.getTimeSeconds());
        if (timespan < this.params.targetTimespan / 4) {
            timespan = this.params.targetTimespan / 4;
        }
        if (timespan > this.params.targetTimespan * 4) {
            timespan = this.params.targetTimespan * 4;
        }
        BigInteger newDifficulty = Utils.decodeCompactBits(blockIntervalAgo.getDifficultyTarget());
        newDifficulty = newDifficulty.multiply(BigInteger.valueOf(timespan));
        if ((newDifficulty = newDifficulty.divide(BigInteger.valueOf(this.params.targetTimespan))).compareTo(this.params.proofOfWorkLimit) > 0) {
            log.debug("Difficulty hit proof of work limit: {}", (Object)newDifficulty.toString(16));
            newDifficulty = this.params.proofOfWorkLimit;
        }
        int accuracyBytes = (int)(next.getDifficultyTarget() >>> 24) - 3;
        BigInteger receivedDifficulty = next.getDifficultyTargetAsInteger();
        BigInteger mask = BigInteger.valueOf(0xFFFFFFL).shiftLeft(accuracyBytes * 8);
        if ((newDifficulty = newDifficulty.and(mask)).compareTo(receivedDifficulty) != 0) {
            throw new VerificationException("Network provided difficulty bits do not match what was calculated: " + receivedDifficulty.toString(16) + " vs " + newDifficulty.toString(16));
        }
    }

    private void scanTransactions(Block block, HashMap<Wallet, List<Transaction>> walletToTxMap) throws VerificationException {
        for (Transaction tx : block.transactions) {
            try {
                for (Wallet wallet : this.wallets) {
                    boolean shouldReceive;
                    if (tx.isCoinBase() || !(shouldReceive = wallet.isTransactionRelevant(tx, true))) continue;
                    List<Transaction> txList = walletToTxMap.get(wallet);
                    if (txList == null) {
                        txList = new LinkedList<Transaction>();
                        walletToTxMap.put(wallet, txList);
                    }
                    txList.add(tx);
                }
            }
            catch (ScriptException e) {
                log.warn("Failed to parse a script: " + e.toString());
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public StoredBlock getChainHead() {
        Object object = this.chainHeadLock;
        synchronized (object) {
            return this.chainHead;
        }
    }

    synchronized Block getUnconnectedBlock() {
        if (this.unconnectedBlocks.size() == 0) {
            return null;
        }
        return this.unconnectedBlocks.get(this.unconnectedBlocks.size() - 1);
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    public static enum NewBlockType {
        BEST_CHAIN,
        SIDE_CHAIN;

    }
}

