/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.AbstractPeerEventListener;
import com.google.bitcoin.core.Address;
import com.google.bitcoin.core.BlockChain;
import com.google.bitcoin.core.ECKey;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.Peer;
import com.google.bitcoin.core.PeerEventListener;
import com.google.bitcoin.core.PeerGroup;
import com.google.bitcoin.core.ScriptException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.StoredBlock;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.TransactionConfidence;
import com.google.bitcoin.core.TransactionInput;
import com.google.bitcoin.core.TransactionOutPoint;
import com.google.bitcoin.core.TransactionOutput;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VerificationException;
import com.google.bitcoin.core.WalletEventListener;
import com.google.bitcoin.core.WalletTransaction;
import com.google.bitcoin.utils.EventListenerInvoker;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class Wallet
implements Serializable {
    private static final Logger log = LoggerFactory.getLogger(Wallet.class);
    private static final long serialVersionUID = 2L;
    final Map<Sha256Hash, Transaction> pending;
    final Map<Sha256Hash, Transaction> unspent;
    final Map<Sha256Hash, Transaction> spent;
    private Map<Sha256Hash, Transaction> inactive;
    private Map<Sha256Hash, Transaction> dead;
    public final ArrayList<ECKey> keychain;
    private final NetworkParameters params;
    private boolean hasTransactionConfidences;
    private transient ArrayList<WalletEventListener> eventListeners;
    private transient PeerEventListener peerEventListener;

    public Wallet(NetworkParameters params) {
        this.params = params;
        this.keychain = new ArrayList();
        this.unspent = new HashMap<Sha256Hash, Transaction>();
        this.spent = new HashMap<Sha256Hash, Transaction>();
        this.inactive = new HashMap<Sha256Hash, Transaction>();
        this.pending = new HashMap<Sha256Hash, Transaction>();
        this.dead = new HashMap<Sha256Hash, Transaction>();
        this.eventListeners = new ArrayList();
        this.hasTransactionConfidences = true;
    }

    public NetworkParameters getNetworkParameters() {
        return this.params;
    }

    public synchronized Iterable<ECKey> getKeys() {
        return new ArrayList<ECKey>(this.keychain);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void saveToFile(File f) throws IOException {
        FileOutputStream stream = null;
        try {
            stream = new FileOutputStream(f);
            this.saveToFileStream(stream);
        }
        finally {
            if (stream != null) {
                stream.close();
            }
        }
    }

    public synchronized void saveToFileStream(OutputStream f) throws IOException {
        ObjectOutputStream oos = new ObjectOutputStream(f);
        oos.writeObject(this);
        oos.close();
    }

    public NetworkParameters getParams() {
        return this.params;
    }

    public static Wallet loadFromFile(File f) throws IOException {
        return Wallet.loadFromFileStream(new FileInputStream(f));
    }

    private boolean isConsistent() {
        HashSet<Transaction> pendingInactive = new HashSet<Transaction>();
        pendingInactive.addAll(this.pending.values());
        pendingInactive.addAll(this.inactive.values());
        return this.getTransactions(true, true).size() == this.unspent.size() + this.spent.size() + pendingInactive.size() + this.dead.size();
    }

    public static Wallet loadFromFileStream(InputStream f) throws IOException {
        ObjectInputStream ois = null;
        try {
            ois = new ObjectInputStream(f);
            Wallet wallet = (Wallet)ois.readObject();
            return wallet;
        }
        catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
        finally {
            if (ois != null) {
                ois.close();
            }
        }
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        this.eventListeners = new ArrayList();
        this.maybeMigrateToTransactionConfidences();
    }

    private void maybeMigrateToTransactionConfidences() {
        if (this.hasTransactionConfidences) {
            return;
        }
        LinkedList<Transaction> transactions = new LinkedList<Transaction>();
        transactions.addAll(this.unspent.values());
        transactions.addAll(this.spent.values());
        for (Transaction tx : transactions) {
            TransactionConfidence confidence = tx.getConfidence();
            confidence.setConfidenceType(TransactionConfidence.ConfidenceType.BUILDING);
            Set<StoredBlock> appearsIn = tx.appearsIn;
            if (appearsIn == null) continue;
            int minHeight = Integer.MAX_VALUE;
            for (StoredBlock block : appearsIn) {
                minHeight = Math.min(minHeight, block.getHeight());
            }
            confidence.setAppearedAtChainHeight(minHeight);
        }
        for (Transaction tx : this.pending.values()) {
            tx.getConfidence().setConfidenceType(TransactionConfidence.ConfidenceType.NOT_SEEN_IN_CHAIN);
        }
        for (Transaction tx : this.inactive.values()) {
            tx.getConfidence().setConfidenceType(TransactionConfidence.ConfidenceType.NOT_IN_BEST_CHAIN);
        }
        for (Transaction tx : this.dead.values()) {
            tx.getConfidence().setConfidenceType(TransactionConfidence.ConfidenceType.OVERRIDDEN_BY_DOUBLE_SPEND);
        }
        this.hasTransactionConfidences = true;
    }

    public synchronized void receiveFromBlock(Transaction tx, StoredBlock block, BlockChain.NewBlockType blockType) throws VerificationException, ScriptException {
        this.receive(tx, block, blockType, false);
    }

    public synchronized void receivePending(Transaction tx) throws VerificationException, ScriptException {
        TransactionConfidence.ConfidenceType currentConfidence;
        EnumSet<WalletTransaction.Pool> containingPools = this.getContainingPools(tx);
        if (!containingPools.equals(EnumSet.noneOf(WalletTransaction.Pool.class))) {
            log.info("Received tx we already saw in a block or created ourselves: " + tx.getHashAsString());
            return;
        }
        if (!this.isTransactionRelevant(tx, true)) {
            log.debug("Received tx that isn't relevant to this wallet, discarding.");
            return;
        }
        BigInteger valueSentToMe = tx.getValueSentToMe(this);
        BigInteger valueSentFromMe = tx.getValueSentFromMe(this);
        if (log.isInfoEnabled()) {
            log.info(String.format("Received a pending transaction %s that spends %s BTC from our own wallet, and sends us %s BTC", tx.getHashAsString(), Utils.bitcoinValueToFriendlyString(valueSentFromMe), Utils.bitcoinValueToFriendlyString(valueSentToMe)));
        }
        if ((currentConfidence = tx.getConfidence().getConfidenceType()) == TransactionConfidence.ConfidenceType.UNKNOWN) {
            tx.getConfidence().setConfidenceType(TransactionConfidence.ConfidenceType.NOT_SEEN_IN_CHAIN);
            this.invokeOnTransactionConfidenceChanged(tx);
        }
        this.commitTx(tx);
    }

    private void invokeOnCoinsReceived(final Transaction tx, final BigInteger balance, final BigInteger newBalance) {
        EventListenerInvoker.invoke(this.eventListeners, new EventListenerInvoker<WalletEventListener>(){

            @Override
            public void invoke(WalletEventListener listener) {
                listener.onCoinsReceived(Wallet.this, tx, balance, newBalance);
            }
        });
    }

    private void invokeOnCoinsSent(final Transaction tx, final BigInteger prevBalance, final BigInteger newBalance) {
        EventListenerInvoker.invoke(this.eventListeners, new EventListenerInvoker<WalletEventListener>(){

            @Override
            public void invoke(WalletEventListener listener) {
                listener.onCoinsSent(Wallet.this, tx, prevBalance, newBalance);
            }
        });
    }

    public synchronized boolean isTransactionRelevant(Transaction tx, boolean includeDoubleSpending) throws ScriptException {
        return tx.getValueSentFromMe(this).compareTo(BigInteger.ZERO) > 0 || tx.getValueSentToMe(this).compareTo(BigInteger.ZERO) > 0 || includeDoubleSpending && this.findDoubleSpendAgainstPending(tx) != null;
    }

    private Transaction findDoubleSpendAgainstPending(Transaction tx) {
        HashSet<TransactionOutPoint> outpoints = new HashSet<TransactionOutPoint>();
        for (TransactionInput input : tx.getInputs()) {
            outpoints.add(input.getOutpoint());
        }
        for (Transaction p : this.pending.values()) {
            for (TransactionInput input : p.getInputs()) {
                if (!outpoints.contains(input.getOutpoint())) continue;
                return p;
            }
        }
        return null;
    }

    private synchronized void receive(Transaction tx, StoredBlock block, BlockChain.NewBlockType blockType, boolean reorg) throws VerificationException, ScriptException {
        boolean wasPending;
        Transaction wtx;
        BigInteger prevBalance = this.getBalance();
        Sha256Hash txHash = tx.getHash();
        boolean bestChain = blockType == BlockChain.NewBlockType.BEST_CHAIN;
        boolean sideChain = blockType == BlockChain.NewBlockType.SIDE_CHAIN;
        BigInteger valueSentFromMe = tx.getValueSentFromMe(this);
        BigInteger valueSentToMe = tx.getValueSentToMe(this);
        BigInteger valueDifference = valueSentToMe.subtract(valueSentFromMe);
        if (!reorg) {
            log.info("Received tx{} for {} BTC: {}", new Object[]{sideChain ? " on a side chain" : "", Utils.bitcoinValueToFriendlyString(valueDifference), tx.getHashAsString()});
        }
        if ((wtx = this.pending.remove(txHash)) != null) {
            boolean alreadyPresent;
            tx = wtx;
            log.info("  <-pending");
            if (bestChain) {
                if (valueSentToMe.equals(BigInteger.ZERO)) {
                    log.info("  ->spent");
                    boolean bl = alreadyPresent = this.spent.put(tx.getHash(), tx) != null;
                    assert (!alreadyPresent) : "TX in both pending and spent pools";
                } else {
                    log.info("  ->unspent");
                    boolean bl = alreadyPresent = this.unspent.put(tx.getHash(), tx) != null;
                    assert (!alreadyPresent) : "TX in both pending and unspent pools";
                }
            } else if (sideChain) {
                log.info("  ->inactive");
                boolean bl = alreadyPresent = this.inactive.put(tx.getHash(), tx) != null;
                if (alreadyPresent) {
                    log.info("Saw a transaction be incorporated into multiple independent side chains");
                }
                this.pending.put(tx.getHash(), tx);
            }
        } else if (sideChain) {
            log.info("  ->inactive");
            this.inactive.put(tx.getHash(), tx);
        } else if (bestChain) {
            this.processTxFromBestChain(tx);
        }
        log.info("Balance is now: " + Utils.bitcoinValueToFriendlyString(this.getBalance()));
        if (block != null) {
            tx.setBlockAppearance(block, bestChain);
            this.invokeOnTransactionConfidenceChanged(tx);
        }
        boolean bl = wasPending = wtx != null;
        if (!reorg && bestChain && !wasPending) {
            BigInteger newBalance = this.getBalance();
            int diff = valueDifference.compareTo(BigInteger.ZERO);
            if (diff > 0) {
                this.invokeOnCoinsReceived(tx, prevBalance, newBalance);
            } else if (diff == 0) {
                this.invokeOnCoinsSent(tx, prevBalance, newBalance);
            } else {
                this.invokeOnCoinsSent(tx, prevBalance, newBalance);
            }
        }
        assert (this.isConsistent());
    }

    private void processTxFromBestChain(Transaction tx) throws VerificationException, ScriptException {
        this.updateForSpends(tx, true);
        if (!tx.getValueSentToMe(this).equals(BigInteger.ZERO)) {
            boolean alreadyPresent;
            log.info("  new tx ->unspent");
            boolean bl = alreadyPresent = this.unspent.put(tx.getHash(), tx) != null;
            assert (!alreadyPresent) : "TX was received twice";
        } else if (!tx.getValueSentFromMe(this).equals(BigInteger.ZERO)) {
            boolean alreadyPresent;
            log.info("  new tx ->spent");
            boolean bl = alreadyPresent = this.spent.put(tx.getHash(), tx) != null;
            assert (!alreadyPresent) : "TX was received twice";
        } else {
            Transaction doubleSpend = this.findDoubleSpendAgainstPending(tx);
            if (doubleSpend == null) {
                throw new IllegalStateException("Received an irrelevant tx that was not a double spend.");
            }
            log.warn("Saw double spend from chain override pending tx {}", (Object)doubleSpend.getHashAsString());
            log.warn("  <-pending ->dead");
            this.pending.remove(doubleSpend.getHash());
            this.dead.put(doubleSpend.getHash(), doubleSpend);
            doubleSpend.getConfidence().setOverridingTransaction(tx);
            this.invokeOnTransactionConfidenceChanged(doubleSpend);
        }
    }

    private void updateForSpends(Transaction tx, boolean fromChain) throws VerificationException {
        List<TransactionInput> inputs = tx.getInputs();
        for (int i = 0; i < inputs.size(); ++i) {
            TransactionInput input = inputs.get(i);
            TransactionInput.ConnectionResult result = input.connect(this.unspent, false);
            if (result == TransactionInput.ConnectionResult.NO_SUCH_TX && (result = input.connect(this.spent, false)) == TransactionInput.ConnectionResult.NO_SUCH_TX) continue;
            if (result == TransactionInput.ConnectionResult.ALREADY_SPENT) {
                Transaction doubleSpent = input.getOutpoint().fromTx;
                assert (doubleSpent != null);
                int index = (int)input.getOutpoint().getIndex();
                TransactionOutput output = doubleSpent.getOutputs().get(index);
                TransactionInput spentBy = output.getSpentBy();
                assert (spentBy != null);
                Transaction connected = spentBy.getParentTransaction();
                assert (connected != null);
                if (fromChain) {
                    if (!this.pending.containsKey(connected.getHash())) continue;
                    log.warn("Saw double spend from chain override pending tx {}", (Object)connected.getHashAsString());
                    log.warn("  <-pending ->dead");
                    this.pending.remove(connected.getHash());
                    this.dead.put(connected.getHash(), connected);
                    input.connect(this.unspent, true);
                    connected.getConfidence().setOverridingTransaction(tx);
                    this.invokeOnTransactionConfidenceChanged(connected);
                    continue;
                }
                log.warn("Saw double spend from another pending transaction, ignoring tx {}", (Object)tx.getHashAsString());
                log.warn("  offending input is input {}", (Object)i);
                return;
            }
            if (result != TransactionInput.ConnectionResult.SUCCESS) continue;
            Transaction connected = input.getOutpoint().fromTx;
            this.maybeMoveTxToSpent(connected, "prevtx");
        }
    }

    private void maybeMoveTxToSpent(Transaction tx, String context) {
        if (tx.isEveryOwnedOutputSpent(this) && this.unspent.remove(tx.getHash()) != null) {
            if (log.isInfoEnabled()) {
                log.info("  " + context + " <-unspent");
                log.info("  " + context + " ->spent");
            }
            this.spent.put(tx.getHash(), tx);
        }
    }

    public synchronized void addEventListener(WalletEventListener listener) {
        this.eventListeners.add(listener);
    }

    public synchronized boolean removeEventListener(WalletEventListener listener) {
        return this.eventListeners.remove(listener);
    }

    public synchronized void commitTx(Transaction tx) throws VerificationException {
        assert (!this.pending.containsKey(tx.getHash())) : "commitTx called on the same transaction twice";
        log.info("commitTx of {}", (Object)tx.getHashAsString());
        BigInteger balance = this.getBalance();
        tx.updatedAt = Utils.now();
        this.updateForSpends(tx, false);
        log.info("->pending: {}", (Object)tx.getHashAsString());
        this.pending.put(tx.getHash(), tx);
        try {
            BigInteger valueSentFromMe = tx.getValueSentFromMe(this);
            BigInteger valueSentToMe = tx.getValueSentToMe(this);
            BigInteger newBalance = balance.add(valueSentToMe).subtract(valueSentFromMe);
            if (valueSentToMe.compareTo(BigInteger.ZERO) > 0) {
                this.invokeOnCoinsReceived(tx, balance, newBalance);
            }
            if (valueSentFromMe.compareTo(BigInteger.ZERO) > 0) {
                this.invokeOnCoinsSent(tx, balance, newBalance);
            }
        }
        catch (ScriptException e) {
            throw new RuntimeException(e);
        }
        assert (this.isConsistent());
    }

    public synchronized Set<Transaction> getTransactions(boolean includeDead, boolean includeInactive) {
        HashSet<Transaction> all = new HashSet<Transaction>();
        all.addAll(this.unspent.values());
        all.addAll(this.spent.values());
        all.addAll(this.pending.values());
        if (includeDead) {
            all.addAll(this.dead.values());
        }
        if (includeInactive) {
            all.addAll(this.inactive.values());
        }
        return all;
    }

    public synchronized Iterable<WalletTransaction> getWalletTransactions() {
        HashSet<Transaction> pendingInactive = new HashSet<Transaction>();
        pendingInactive.addAll(this.pending.values());
        pendingInactive.retainAll(this.inactive.values());
        HashSet<Transaction> onlyPending = new HashSet<Transaction>();
        HashSet<Transaction> onlyInactive = new HashSet<Transaction>();
        onlyPending.addAll(this.pending.values());
        onlyPending.removeAll(pendingInactive);
        onlyInactive.addAll(this.inactive.values());
        onlyInactive.removeAll(pendingInactive);
        HashSet<WalletTransaction> all = new HashSet<WalletTransaction>();
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.UNSPENT, this.unspent.values());
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.SPENT, this.spent.values());
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.DEAD, this.dead.values());
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.PENDING, onlyPending);
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.INACTIVE, onlyInactive);
        Wallet.addWalletTransactionsToSet(all, WalletTransaction.Pool.PENDING_INACTIVE, pendingInactive);
        return all;
    }

    private static synchronized void addWalletTransactionsToSet(Set<WalletTransaction> txs, WalletTransaction.Pool poolType, Collection<Transaction> pool) {
        for (Transaction tx : pool) {
            txs.add(new WalletTransaction(poolType, tx));
        }
    }

    public synchronized void addWalletTransaction(WalletTransaction wtx) {
        switch (wtx.getPool()) {
            case UNSPENT: {
                this.unspent.put(wtx.getTransaction().getHash(), wtx.getTransaction());
                break;
            }
            case SPENT: {
                this.spent.put(wtx.getTransaction().getHash(), wtx.getTransaction());
                break;
            }
            case PENDING: {
                this.pending.put(wtx.getTransaction().getHash(), wtx.getTransaction());
                break;
            }
            case DEAD: {
                this.dead.put(wtx.getTransaction().getHash(), wtx.getTransaction());
                break;
            }
            case INACTIVE: {
                this.inactive.put(wtx.getTransaction().getHash(), wtx.getTransaction());
                break;
            }
            case PENDING_INACTIVE: {
                this.pending.put(wtx.getTransaction().getHash(), wtx.getTransaction());
                this.inactive.put(wtx.getTransaction().getHash(), wtx.getTransaction());
                break;
            }
            default: {
                throw new RuntimeException("Unknown wallet transaction type " + (Object)((Object)wtx.getPool()));
            }
        }
    }

    public List<Transaction> getTransactionsByTime() {
        return this.getRecentTransactions(0, false);
    }

    public synchronized List<Transaction> getRecentTransactions(int numTransactions, boolean includeDead) {
        assert (numTransactions >= 0);
        int size = this.getPoolSize(WalletTransaction.Pool.UNSPENT) + this.getPoolSize(WalletTransaction.Pool.SPENT) + this.getPoolSize(WalletTransaction.Pool.PENDING);
        if (numTransactions > size || numTransactions == 0) {
            numTransactions = size;
        }
        ArrayList<Transaction> all = new ArrayList<Transaction>(this.getTransactions(includeDead, false));
        Collections.sort(all, Collections.reverseOrder(new Comparator<Transaction>(){

            @Override
            public int compare(Transaction t1, Transaction t2) {
                return t1.getUpdateTime().compareTo(t2.getUpdateTime());
            }
        }));
        if (numTransactions == all.size()) {
            return all;
        }
        all.subList(numTransactions, all.size()).clear();
        return all;
    }

    public synchronized Transaction getTransaction(Sha256Hash hash) {
        Transaction tx = this.pending.get(hash);
        if (tx != null) {
            return tx;
        }
        tx = this.unspent.get(hash);
        if (tx != null) {
            return tx;
        }
        tx = this.spent.get(hash);
        if (tx != null) {
            return tx;
        }
        tx = this.inactive.get(hash);
        if (tx != null) {
            return tx;
        }
        tx = this.dead.get(hash);
        if (tx != null) {
            return tx;
        }
        return null;
    }

    public synchronized void clearTransactions(int fromHeight) {
        if (fromHeight != 0) {
            throw new UnsupportedOperationException();
        }
        this.unspent.clear();
        this.spent.clear();
        this.pending.clear();
        this.inactive.clear();
        this.dead.clear();
    }

    synchronized EnumSet<WalletTransaction.Pool> getContainingPools(Transaction tx) {
        EnumSet<WalletTransaction.Pool> result = EnumSet.noneOf(WalletTransaction.Pool.class);
        Sha256Hash txHash = tx.getHash();
        if (this.unspent.containsKey(txHash)) {
            result.add(WalletTransaction.Pool.UNSPENT);
        }
        if (this.spent.containsKey(txHash)) {
            result.add(WalletTransaction.Pool.SPENT);
        }
        if (this.pending.containsKey(txHash)) {
            result.add(WalletTransaction.Pool.PENDING);
        }
        if (this.inactive.containsKey(txHash)) {
            result.add(WalletTransaction.Pool.INACTIVE);
        }
        if (this.dead.containsKey(txHash)) {
            result.add(WalletTransaction.Pool.DEAD);
        }
        return result;
    }

    synchronized int getPoolSize(WalletTransaction.Pool pool) {
        switch (pool) {
            case UNSPENT: {
                return this.unspent.size();
            }
            case SPENT: {
                return this.spent.size();
            }
            case PENDING: {
                return this.pending.size();
            }
            case INACTIVE: {
                return this.inactive.size();
            }
            case DEAD: {
                return this.dead.size();
            }
            case ALL: {
                return this.unspent.size() + this.spent.size() + this.pending.size() + this.inactive.size() + this.dead.size();
            }
        }
        throw new RuntimeException("Unreachable");
    }

    public synchronized Transaction createSend(Address address, BigInteger nanocoins) {
        return this.createSend(address, nanocoins, this.getChangeAddress());
    }

    public synchronized Transaction sendCoinsOffline(Address to, BigInteger nanocoins) {
        Transaction tx = this.createSend(to, nanocoins);
        if (tx == null) {
            return null;
        }
        try {
            this.commitTx(tx);
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
        return tx;
    }

    public synchronized Transaction sendCoinsAsync(PeerGroup peerGroup, Address to, BigInteger nanocoins) throws IOException {
        Transaction tx = this.sendCoinsOffline(to, nanocoins);
        if (tx == null) {
            return null;
        }
        peerGroup.broadcastTransaction(tx);
        return tx;
    }

    public synchronized Transaction sendCoins(PeerGroup peerGroup, Address to, BigInteger nanocoins) {
        Transaction tx = this.sendCoinsOffline(to, nanocoins);
        if (tx == null) {
            return null;
        }
        try {
            return peerGroup.broadcastTransaction(tx).get();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    public synchronized Transaction sendCoins(Peer peer, Address to, BigInteger nanocoins) throws IOException {
        Transaction tx = this.createSend(to, nanocoins);
        if (tx == null) {
            return null;
        }
        try {
            this.commitTx(tx);
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
        peer.sendMessage(tx);
        return tx;
    }

    public synchronized Transaction createSend(Address address, BigInteger nanocoins, Address changeAddress) {
        log.info("Creating send tx to " + address.toString() + " for " + Utils.bitcoinValueToFriendlyString(nanocoins));
        Transaction sendTx = new Transaction(this.params);
        sendTx.addOutput(nanocoins, address);
        if (this.completeTx(sendTx, changeAddress)) {
            return sendTx;
        }
        return null;
    }

    public synchronized boolean completeTx(Transaction sendTx, Address changeAddress) {
        BigInteger nanocoins = BigInteger.ZERO;
        for (TransactionOutput output : sendTx.getOutputs()) {
            nanocoins = nanocoins.add(output.getValue());
        }
        log.info("Completing send tx with {} outputs totalling {}", (Object)sendTx.getOutputs().size(), (Object)Utils.bitcoinValueToFriendlyString(nanocoins));
        BigInteger valueGathered = BigInteger.ZERO;
        LinkedList<TransactionOutput> gathered = new LinkedList<TransactionOutput>();
        for (Transaction tx : this.unspent.values()) {
            for (TransactionOutput output : tx.getOutputs()) {
                if (!output.isAvailableForSpending() || !output.isMine(this)) continue;
                gathered.add(output);
                valueGathered = valueGathered.add(output.getValue());
            }
            if (valueGathered.compareTo(nanocoins) < 0) continue;
            break;
        }
        if (valueGathered.compareTo(nanocoins) < 0) {
            log.info("Insufficient value in wallet for send, missing " + Utils.bitcoinValueToFriendlyString(nanocoins.subtract(valueGathered)));
            return false;
        }
        assert (gathered.size() > 0);
        sendTx.getConfidence().setConfidenceType(TransactionConfidence.ConfidenceType.NOT_SEEN_IN_CHAIN);
        BigInteger change = valueGathered.subtract(nanocoins);
        if (change.compareTo(BigInteger.ZERO) > 0) {
            log.info("  with " + Utils.bitcoinValueToFriendlyString(change) + " coins change");
            sendTx.addOutput(new TransactionOutput(this.params, sendTx, change, changeAddress));
        }
        for (TransactionOutput output : gathered) {
            sendTx.addInput(output);
        }
        try {
            sendTx.signInputs(Transaction.SigHash.ALL, this);
        }
        catch (ScriptException e) {
            throw new RuntimeException(e);
        }
        log.info("  completed {}", (Object)sendTx.getHashAsString());
        return true;
    }

    public synchronized boolean completeTx(Transaction sendTx) {
        return this.completeTx(sendTx, this.getChangeAddress());
    }

    synchronized Address getChangeAddress() {
        assert (this.keychain.size() > 0) : "Can't send value without an address to use for receiving change";
        ECKey first = this.keychain.get(0);
        return first.toAddress(this.params);
    }

    public synchronized void addKey(ECKey key) {
        assert (!this.keychain.contains(key));
        this.keychain.add(key);
    }

    public synchronized ECKey findKeyFromPubHash(byte[] pubkeyHash) {
        for (ECKey key : this.keychain) {
            if (!Arrays.equals(key.getPubKeyHash(), pubkeyHash)) continue;
            return key;
        }
        return null;
    }

    public synchronized boolean isPubKeyHashMine(byte[] pubkeyHash) {
        return this.findKeyFromPubHash(pubkeyHash) != null;
    }

    public synchronized ECKey findKeyFromPubKey(byte[] pubkey) {
        for (ECKey key : this.keychain) {
            if (!Arrays.equals(key.getPubKey(), pubkey)) continue;
            return key;
        }
        return null;
    }

    public synchronized boolean isPubKeyMine(byte[] pubkey) {
        return this.findKeyFromPubKey(pubkey) != null;
    }

    public synchronized BigInteger getBalance() {
        return this.getBalance(BalanceType.AVAILABLE);
    }

    public synchronized BigInteger getBalance(BalanceType balanceType) {
        BigInteger available = BigInteger.ZERO;
        for (Transaction tx : this.unspent.values()) {
            for (TransactionOutput output : tx.getOutputs()) {
                if (!output.isMine(this) || !output.isAvailableForSpending()) continue;
                available = available.add(output.getValue());
            }
        }
        if (balanceType == BalanceType.AVAILABLE) {
            return available;
        }
        assert (balanceType == BalanceType.ESTIMATED);
        BigInteger estimated = available;
        for (Transaction tx : this.pending.values()) {
            for (TransactionOutput output : tx.getOutputs()) {
                if (!output.isMine(this)) continue;
                estimated = estimated.add(output.getValue());
            }
        }
        return estimated;
    }

    public synchronized String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append(String.format("Wallet containing %s BTC in:\n", Utils.bitcoinValueToFriendlyString(this.getBalance())));
        builder.append(String.format("  %d unspent transactions\n", this.unspent.size()));
        builder.append(String.format("  %d spent transactions\n", this.spent.size()));
        builder.append(String.format("  %d pending transactions\n", this.pending.size()));
        builder.append(String.format("  %d inactive transactions\n", this.inactive.size()));
        builder.append(String.format("  %d dead transactions\n", this.dead.size()));
        builder.append("\nKeys:\n");
        for (ECKey key : this.keychain) {
            builder.append("  addr:");
            builder.append(key.toAddress(this.params));
            builder.append(" ");
            builder.append(key.toString());
            builder.append("\n");
        }
        if (this.unspent.size() > 0) {
            builder.append("\nUNSPENT:\n");
            this.toStringHelper(builder, this.unspent);
        }
        if (this.spent.size() > 0) {
            builder.append("\nSPENT:\n");
            this.toStringHelper(builder, this.spent);
        }
        if (this.pending.size() > 0) {
            builder.append("\nPENDING:\n");
            this.toStringHelper(builder, this.pending);
        }
        if (this.inactive.size() > 0) {
            builder.append("\nINACTIVE:\n");
            this.toStringHelper(builder, this.inactive);
        }
        if (this.dead.size() > 0) {
            builder.append("\nDEAD:\n");
            this.toStringHelper(builder, this.dead);
        }
        return builder.toString();
    }

    private void toStringHelper(StringBuilder builder, Map<Sha256Hash, Transaction> transactionMap) {
        for (Transaction tx : transactionMap.values()) {
            try {
                builder.append("Sends ");
                builder.append(Utils.bitcoinValueToFriendlyString(tx.getValueSentFromMe(this)));
                builder.append(" and receives ");
                builder.append(Utils.bitcoinValueToFriendlyString(tx.getValueSentToMe(this)));
                builder.append(", total value ");
                builder.append(Utils.bitcoinValueToFriendlyString(tx.getValue(this)));
                builder.append(".\n");
            }
            catch (ScriptException e) {
                // empty catch block
            }
            builder.append(tx);
        }
    }

    synchronized void reorganize(List<StoredBlock> oldBlocks, List<StoredBlock> newBlocks) throws VerificationException {
        ArrayList<Sha256Hash> oldBlockHashes = new ArrayList<Sha256Hash>(oldBlocks.size());
        ArrayList<Sha256Hash> newBlockHashes = new ArrayList<Sha256Hash>(newBlocks.size());
        log.info("Old part of chain (top to bottom):");
        for (StoredBlock b : oldBlocks) {
            log.info("  {}", (Object)b.getHeader().getHashAsString());
            oldBlockHashes.add(b.getHeader().getHash());
        }
        log.info("New part of chain (top to bottom):");
        for (StoredBlock b : newBlocks) {
            log.info("  {}", (Object)b.getHeader().getHashAsString());
            newBlockHashes.add(b.getHeader().getHash());
        }
        HashMap<Sha256Hash, Transaction> oldChainTransactions = new HashMap<Sha256Hash, Transaction>();
        HashMap<Sha256Hash, Transaction> onlyOldChainTransactions = new HashMap<Sha256Hash, Transaction>();
        HashMap<Sha256Hash, Transaction> newChainTransactions = new HashMap<Sha256Hash, Transaction>();
        HashMap<Sha256Hash, Transaction> commonChainTransactions = new HashMap<Sha256Hash, Transaction>();
        HashMap<Sha256Hash, Transaction> all = new HashMap<Sha256Hash, Transaction>();
        all.putAll(this.unspent);
        all.putAll(this.spent);
        all.putAll(this.inactive);
        for (Transaction tx : all.values()) {
            boolean alreadyPresent;
            boolean inCommonSection;
            Collection<Sha256Hash> appearsIn = tx.getAppearsInHashes();
            assert (appearsIn != null);
            boolean inOldSection = !Collections.disjoint(appearsIn, oldBlockHashes);
            boolean inNewSection = !Collections.disjoint(appearsIn, newBlockHashes);
            boolean bl = inCommonSection = !inNewSection && !inOldSection;
            if (inCommonSection) {
                boolean bl2 = alreadyPresent = commonChainTransactions.put(tx.getHash(), tx) != null;
                assert (!alreadyPresent) : "Transaction appears twice in common chain segment";
                continue;
            }
            if (inOldSection) {
                boolean bl3 = alreadyPresent = oldChainTransactions.put(tx.getHash(), tx) != null;
                assert (!alreadyPresent) : "Transaction appears twice in old chain segment";
                if (!inNewSection) {
                    boolean bl4 = alreadyPresent = onlyOldChainTransactions.put(tx.getHash(), tx) != null;
                    assert (!alreadyPresent) : "Transaction appears twice in only-old map";
                }
            }
            if (!inNewSection) continue;
            boolean bl5 = alreadyPresent = newChainTransactions.put(tx.getHash(), tx) != null;
            assert (!alreadyPresent) : "Transaction appears twice in new chain segment";
        }
        boolean affectedUs = !((Object)oldChainTransactions).equals(newChainTransactions);
        log.info(affectedUs ? "Re-org affected our transactions" : "Re-org had no effect on our transactions");
        if (!affectedUs) {
            return;
        }
        for (Transaction tx : onlyOldChainTransactions.values()) {
            log.info("  Only Old: {}", (Object)tx.getHashAsString());
        }
        for (Transaction tx : oldChainTransactions.values()) {
            log.info("  Old: {}", (Object)tx.getHashAsString());
        }
        for (Transaction tx : newChainTransactions.values()) {
            log.info("  New: {}", (Object)tx.getHashAsString());
        }
        for (Transaction tx : all.values()) {
            tx.disconnectInputs();
        }
        for (Transaction tx : this.pending.values()) {
            tx.disconnectInputs();
        }
        for (Transaction tx : commonChainTransactions.values()) {
            TransactionInput badInput = tx.connectForReorganize(all);
            assert (badInput == null) : "Failed to connect " + tx.getHashAsString() + ", " + badInput.toString();
        }
        log.info("Moving transactions");
        this.unspent.clear();
        this.spent.clear();
        this.inactive.clear();
        for (Transaction tx : commonChainTransactions.values()) {
            int unspentOutputs = 0;
            for (TransactionOutput output : tx.getOutputs()) {
                if (!output.isAvailableForSpending()) continue;
                ++unspentOutputs;
            }
            if (unspentOutputs > 0) {
                log.info("  TX {}: ->unspent", (Object)tx.getHashAsString());
                this.unspent.put(tx.getHash(), tx);
                continue;
            }
            log.info("  TX {}: ->spent", (Object)tx.getHashAsString());
            this.spent.put(tx.getHash(), tx);
        }
        for (Transaction tx : onlyOldChainTransactions.values()) {
            tx.notifyNotOnBestChain();
        }
        Collections.reverse(newBlocks);
        for (StoredBlock b : newBlocks) {
            log.info("Replaying block {}", (Object)b.getHeader().getHashAsString());
            HashSet<Transaction> txns = new HashSet<Transaction>();
            Sha256Hash blockHash = b.getHeader().getHash();
            for (Transaction tx : newChainTransactions.values()) {
                if (!tx.getAppearsInHashes().contains(blockHash)) continue;
                txns.add(tx);
                log.info("  containing tx {}", (Object)tx.getHashAsString());
            }
            for (Transaction t : txns) {
                try {
                    this.receive(t, b, BlockChain.NewBlockType.BEST_CHAIN, true);
                }
                catch (ScriptException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        HashMap<Sha256Hash, Transaction> pool = new HashMap<Sha256Hash, Transaction>();
        pool.putAll(this.unspent);
        pool.putAll(this.spent);
        pool.putAll(this.pending);
        HashMap<Sha256Hash, Transaction> toReprocess = new HashMap<Sha256Hash, Transaction>();
        toReprocess.putAll(onlyOldChainTransactions);
        toReprocess.putAll(this.pending);
        log.info("Reprocessing transactions not in new best chain:");
        for (Transaction tx : this.dead.values()) {
            this.reprocessTxAfterReorg(pool, tx);
        }
        for (Transaction tx : toReprocess.values()) {
            this.reprocessTxAfterReorg(pool, tx);
        }
        log.info("post-reorg balance is {}", (Object)Utils.bitcoinValueToFriendlyString(this.getBalance()));
        EventListenerInvoker.invoke(this.eventListeners, new EventListenerInvoker<WalletEventListener>(){

            @Override
            public void invoke(WalletEventListener listener) {
                listener.onReorganize(Wallet.this);
            }
        });
        assert (this.isConsistent());
    }

    private void reprocessTxAfterReorg(Map<Sha256Hash, Transaction> pool, Transaction tx) {
        log.info("TX {}", (Object)tx.getHashAsString());
        int numInputs = tx.getInputs().size();
        int noSuchTx = 0;
        int success = 0;
        boolean isDead = false;
        for (TransactionInput input : tx.getInputs()) {
            if (input.isCoinBase()) {
                ++noSuchTx;
                continue;
            }
            TransactionInput.ConnectionResult result = input.connect(pool, false);
            if (result == TransactionInput.ConnectionResult.SUCCESS) {
                ++success;
                continue;
            }
            if (result == TransactionInput.ConnectionResult.NO_SUCH_TX) {
                ++noSuchTx;
                continue;
            }
            if (result != TransactionInput.ConnectionResult.ALREADY_SPENT) continue;
            isDead = true;
            log.info("   ->dead, will not confirm now unless there's another re-org", (Object)tx.getHashAsString());
            TransactionOutput doubleSpent = input.getConnectedOutput(pool);
            Transaction replacement = doubleSpent.getSpentBy().getParentTransaction();
            this.dead.put(tx.getHash(), tx);
            this.pending.remove(tx.getHash());
            tx.getConfidence().setOverridingTransaction(replacement);
            this.invokeOnTransactionConfidenceChanged(tx);
            break;
        }
        if (isDead) {
            return;
        }
        if (noSuchTx == numInputs) {
            log.info("   ->inactive", (Object)tx.getHashAsString());
            this.inactive.put(tx.getHash(), tx);
        } else if (success == numInputs - noSuchTx) {
            log.info("   ->pending", (Object)tx.getHashAsString());
            this.pending.put(tx.getHash(), tx);
            this.dead.remove(tx.getHash());
        }
    }

    private void invokeOnTransactionConfidenceChanged(final Transaction tx) {
        EventListenerInvoker.invoke(this.eventListeners, new EventListenerInvoker<WalletEventListener>(){

            @Override
            public void invoke(WalletEventListener listener) {
                listener.onTransactionConfidenceChanged(Wallet.this, tx);
            }
        });
    }

    public synchronized Collection<Transaction> getPendingTransactions() {
        return Collections.unmodifiableCollection(this.pending.values());
    }

    public synchronized long getEarliestKeyCreationTime() {
        if (this.keychain.size() == 0) {
            return Utils.now().getTime() / 1000L;
        }
        long earliestTime = Long.MAX_VALUE;
        for (ECKey key : this.keychain) {
            earliestTime = Math.min(key.getCreationTimeSeconds(), earliestTime);
        }
        return earliestTime;
    }

    public synchronized PeerEventListener getPeerEventListener() {
        if (this.peerEventListener == null) {
            this.peerEventListener = new AbstractPeerEventListener(){

                public void onTransaction(Peer peer, Transaction t) {
                    try {
                        Wallet.this.receivePending(t);
                    }
                    catch (VerificationException e) {
                        log.warn("Received broadcast transaction that does not validate: {}", (Object)t);
                        log.warn("VerificationException caught", (Throwable)e);
                    }
                    catch (ScriptException e) {
                        log.warn("Received broadcast transaction with not understood scripts: {}", (Object)t);
                        log.warn("ScriptException caught", (Throwable)e);
                    }
                }
            };
        }
        return this.peerEventListener;
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    public static enum BalanceType {
        ESTIMATED,
        AVAILABLE;

    }
}

