/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.store;

import com.google.bitcoin.core.Block;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.ProtocolException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.StoredBlock;
import com.google.bitcoin.core.VerificationException;
import com.google.bitcoin.store.BlockStore;
import com.google.bitcoin.store.BlockStoreException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DiskBlockStore
implements BlockStore {
    private static final Logger log = LoggerFactory.getLogger(DiskBlockStore.class);
    private RandomAccessFile file;
    private Map<Sha256Hash, StoredBlock> blockMap;
    private Sha256Hash chainHead;
    private NetworkParameters params;

    public DiskBlockStore(NetworkParameters params, File theFile) throws BlockStoreException {
        this.params = params;
        this.blockMap = new HashMap<Sha256Hash, StoredBlock>();
        try {
            this.file = new RandomAccessFile(theFile, "rwd");
            this.load(theFile);
        }
        catch (IOException e) {
            log.error("failed to load block store from file", (Throwable)e);
            this.createNewStore(params);
        }
    }

    private void createNewStore(NetworkParameters params) throws BlockStoreException {
        this.blockMap.clear();
        try {
            this.file.write(1);
        }
        catch (IOException e1) {
            throw new BlockStoreException(e1);
        }
        try {
            Block genesis = params.genesisBlock.cloneAsHeader();
            StoredBlock storedGenesis = new StoredBlock(genesis, genesis.getWork(), 0);
            this.chainHead = storedGenesis.getHeader().getHash();
            this.file.write(this.chainHead.getBytes());
            this.put(storedGenesis);
        }
        catch (VerificationException e1) {
            throw new RuntimeException(e1);
        }
        catch (IOException e) {
            throw new BlockStoreException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void load(File theFile) throws IOException, BlockStoreException {
        log.info("Reading block store from {}", (Object)theFile);
        int version = this.file.read();
        if (version == -1) {
            throw new FileNotFoundException(theFile.getName() + " is empty");
        }
        if (version != 1) {
            throw new BlockStoreException("Bad version number: " + version);
        }
        byte[] chainHeadHash = new byte[32];
        if (this.file.read(chainHeadHash) < chainHeadHash.length) {
            throw new BlockStoreException("Truncated block store: cannot read chain head hash");
        }
        this.chainHead = new Sha256Hash(chainHeadHash);
        log.info("Read chain head from disk: {}", (Object)this.chainHead);
        long now = System.currentTimeMillis();
        byte[] headerBytes = new byte[80];
        try {
            int read;
            while ((read = this.file.read(headerBytes)) != -1) {
                StoredBlock s;
                if (read < headerBytes.length) {
                    throw new BlockStoreException("Truncated block store: partial block read");
                }
                Block b = new Block(this.params, headerBytes);
                StoredBlock prev = this.get(b.getPrevBlockHash());
                if (prev == null) {
                    if (!b.equals(this.params.genesisBlock)) throw new BlockStoreException("Could not connect " + b.getHash().toString() + " to " + b.getPrevBlockHash().toString());
                    s = new StoredBlock(this.params.genesisBlock.cloneAsHeader(), this.params.genesisBlock.getWork(), 0);
                } else {
                    b.verifyHeader();
                    s = prev.build(b);
                }
                this.blockMap.put(b.getHash(), s);
            }
        }
        catch (ProtocolException e) {
            throw new BlockStoreException(e);
        }
        catch (VerificationException e) {
            throw new BlockStoreException(e);
        }
        long elapsed = System.currentTimeMillis() - now;
        log.info("Block chain read complete in {}ms", (Object)elapsed);
        return;
    }

    public synchronized void put(StoredBlock block) throws BlockStoreException {
        try {
            Sha256Hash hash = block.getHeader().getHash();
            assert (this.blockMap.get(hash) == null) : "Attempt to insert duplicate";
            byte[] bytes = block.getHeader().bitcoinSerialize();
            this.file.write(bytes);
            this.blockMap.put(hash, block);
        }
        catch (IOException e) {
            throw new BlockStoreException(e);
        }
    }

    public synchronized StoredBlock get(Sha256Hash hash) throws BlockStoreException {
        return this.blockMap.get(hash);
    }

    public synchronized StoredBlock getChainHead() throws BlockStoreException {
        return this.blockMap.get(this.chainHead);
    }

    public synchronized void setChainHead(StoredBlock chainHead) throws BlockStoreException {
        try {
            this.chainHead = chainHead.getHeader().getHash();
            this.file.getChannel().write(ByteBuffer.wrap(this.chainHead.getBytes()), 1L);
        }
        catch (IOException e) {
            throw new BlockStoreException(e);
        }
    }
}

