/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.store;

import com.google.bitcoin.core.ECKey;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.TransactionConfidence;
import com.google.bitcoin.core.TransactionInput;
import com.google.bitcoin.core.TransactionOutPoint;
import com.google.bitcoin.core.TransactionOutput;
import com.google.bitcoin.core.Wallet;
import com.google.bitcoin.core.WalletTransaction;
import com.google.protobuf.ByteString;
import com.google.protobuf.Message;
import com.google.protobuf.TextFormat;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import org.bitcoinj.wallet.Protos;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WalletProtobufSerializer {
    private static final Logger log = LoggerFactory.getLogger(WalletProtobufSerializer.class);
    private Map<ByteString, Transaction> txMap = new HashMap<ByteString, Transaction>();

    private WalletProtobufSerializer() {
    }

    public static void writeWallet(Wallet wallet, OutputStream output) throws IOException {
        Protos.Wallet walletProto = WalletProtobufSerializer.walletToProto(wallet);
        walletProto.writeTo(output);
    }

    public static String walletToText(Wallet wallet) {
        Protos.Wallet walletProto = WalletProtobufSerializer.walletToProto(wallet);
        return TextFormat.printToString((Message)walletProto);
    }

    public static Protos.Wallet walletToProto(Wallet wallet) {
        Protos.Wallet.Builder walletBuilder = Protos.Wallet.newBuilder();
        walletBuilder.setNetworkIdentifier(wallet.getNetworkParameters().getId());
        for (WalletTransaction wtx : wallet.getWalletTransactions()) {
            Protos.Transaction txProto = WalletProtobufSerializer.makeTxProto(wtx);
            walletBuilder.addTransaction(txProto);
        }
        for (ECKey key : wallet.getKeys()) {
            walletBuilder.addKey(Protos.Key.newBuilder().setCreationTimestamp(key.getCreationTimeSeconds() * 1000L).setType(Protos.Key.Type.ORIGINAL).setPrivateKey(ByteString.copyFrom((byte[])key.getPrivKeyBytes())).setPublicKey(ByteString.copyFrom((byte[])key.getPubKey())));
        }
        return walletBuilder.build();
    }

    private static Protos.Transaction makeTxProto(WalletTransaction wtx) {
        Transaction tx = wtx.getTransaction();
        Protos.Transaction.Builder txBuilder = Protos.Transaction.newBuilder();
        txBuilder.setPool(Protos.Transaction.Pool.valueOf(wtx.getPool().getValue())).setHash(ByteString.copyFrom((byte[])tx.getHash().getBytes())).setVersion((int)tx.getVersion());
        if (tx.getUpdateTime() != null) {
            txBuilder.setUpdatedAt(tx.getUpdateTime().getTime());
        }
        if (tx.getLockTime() > 0L) {
            txBuilder.setLockTime((int)tx.getLockTime());
        }
        for (TransactionInput input : tx.getInputs()) {
            Protos.TransactionInput.Builder inputBuilder = Protos.TransactionInput.newBuilder().setScriptBytes(ByteString.copyFrom((byte[])input.getScriptBytes())).setTransactionOutPointHash(ByteString.copyFrom((byte[])input.getOutpoint().getHash().getBytes())).setTransactionOutPointIndex((int)input.getOutpoint().getIndex());
            if (input.hasSequence()) {
                inputBuilder.setSequence((int)input.getSequence());
            }
            txBuilder.addTransactionInput(inputBuilder);
        }
        for (TransactionOutput output : tx.getOutputs()) {
            Protos.TransactionOutput.Builder outputBuilder = Protos.TransactionOutput.newBuilder().setScriptBytes(ByteString.copyFrom((byte[])output.getScriptBytes())).setValue(output.getValue().longValue());
            TransactionInput spentBy = output.getSpentBy();
            if (spentBy != null) {
                Sha256Hash spendingHash = spentBy.getParentTransaction().getHash();
                outputBuilder.setSpentByTransactionHash(WalletProtobufSerializer.hashToByteString(spendingHash)).setSpentByTransactionIndex(spentBy.getParentTransaction().getInputs().indexOf(spentBy));
            }
            txBuilder.addTransactionOutput(outputBuilder);
        }
        if (tx.getAppearsInHashes() != null) {
            for (Sha256Hash hash : tx.getAppearsInHashes()) {
                txBuilder.addBlockHash(WalletProtobufSerializer.hashToByteString(hash));
            }
        }
        if (tx.hasConfidence()) {
            TransactionConfidence confidence = tx.getConfidence();
            Protos.TransactionConfidence.Builder confidenceBuilder = Protos.TransactionConfidence.newBuilder();
            WalletProtobufSerializer.writeConfidence(txBuilder, confidence, confidenceBuilder);
        }
        return txBuilder.build();
    }

    private static void writeConfidence(Protos.Transaction.Builder txBuilder, TransactionConfidence confidence, Protos.TransactionConfidence.Builder confidenceBuilder) {
        confidenceBuilder.setType(Protos.TransactionConfidence.Type.valueOf(confidence.getConfidenceType().getValue()));
        if (confidence.getConfidenceType() == TransactionConfidence.ConfidenceType.BUILDING) {
            confidenceBuilder.setAppearedAtHeight(confidence.getAppearedAtChainHeight());
        }
        if (confidence.getConfidenceType() == TransactionConfidence.ConfidenceType.OVERRIDDEN_BY_DOUBLE_SPEND) {
            Sha256Hash overridingHash = confidence.getOverridingTransaction().getHash();
            confidenceBuilder.setOverridingTransaction(WalletProtobufSerializer.hashToByteString(overridingHash));
        }
        txBuilder.setConfidence(confidenceBuilder);
    }

    private static ByteString hashToByteString(Sha256Hash hash) {
        return ByteString.copyFrom((byte[])hash.getBytes());
    }

    public static Wallet readWallet(InputStream input, NetworkParameters params) throws IOException {
        WalletProtobufSerializer serializer = new WalletProtobufSerializer();
        Protos.Wallet walletProto = Protos.Wallet.parseFrom(input);
        if (!params.getId().equals(walletProto.getNetworkIdentifier())) {
            throw new IllegalArgumentException("Trying to read a wallet with a different network id " + walletProto.getNetworkIdentifier());
        }
        Wallet wallet = new Wallet(params);
        for (Protos.Key keyProto : walletProto.getKeyList()) {
            if (keyProto.getType() != Protos.Key.Type.ORIGINAL) {
                throw new IllegalArgumentException("Unknown key type in wallet");
            }
            if (!keyProto.hasPrivateKey()) {
                throw new IllegalArgumentException("Don't know how to handle pubkey-only keys");
            }
            byte[] pubKey = keyProto.hasPublicKey() ? keyProto.getPublicKey().toByteArray() : null;
            ECKey ecKey = new ECKey(keyProto.getPrivateKey().toByteArray(), pubKey);
            ecKey.setCreationTimeSeconds((keyProto.getCreationTimestamp() + 500L) / 1000L);
            wallet.addKey(ecKey);
        }
        for (Protos.Transaction txProto : walletProto.getTransactionList()) {
            serializer.readTransaction(txProto, params);
        }
        for (Protos.Transaction txProto : walletProto.getTransactionList()) {
            serializer.connectTransactionInputs(txProto, params);
        }
        for (Protos.Transaction txProto : walletProto.getTransactionList()) {
            WalletTransaction wtx = serializer.connectTransactionOutputs(txProto, params);
            wallet.addWalletTransaction(wtx);
        }
        for (Protos.Extension extProto : walletProto.getExtensionList()) {
            if (!extProto.getMandatory()) continue;
            throw new IllegalArgumentException("Did not understand a mandatory extension in the wallet");
        }
        return wallet;
    }

    private void readTransaction(Protos.Transaction txProto, NetworkParameters params) {
        Transaction tx = new Transaction(params, txProto.getVersion(), new Sha256Hash(txProto.getHash().toByteArray()));
        if (txProto.hasUpdatedAt()) {
            tx.setUpdateTime(new Date(txProto.getUpdatedAt()));
        }
        for (Protos.TransactionOutput outputProto : txProto.getTransactionOutputList()) {
            TransactionOutput output = new TransactionOutput(params, tx, BigInteger.valueOf(outputProto.getValue()), outputProto.getScriptBytes().toByteArray());
            tx.addOutput(output);
        }
        if (this.txMap.containsKey(ByteString.copyFrom((byte[])tx.getHash().getBytes()))) {
            throw new RuntimeException("Transaction " + tx.getHashAsString() + " appears twice");
        }
        for (ByteString blockHash : txProto.getBlockHashList()) {
            tx.addBlockAppearance(new Sha256Hash(blockHash.toByteArray()));
        }
        if (txProto.hasLockTime()) {
            tx.setLockTime(txProto.getLockTime());
        }
        this.txMap.put(txProto.getHash(), tx);
    }

    private void connectTransactionInputs(Protos.Transaction txProto, NetworkParameters params) {
        Transaction tx = this.txMap.get(txProto.getHash());
        for (Protos.TransactionInput transactionInput : txProto.getTransactionInputList()) {
            TransactionInput input = new TransactionInput(params, tx, transactionInput.getScriptBytes().toByteArray(), new TransactionOutPoint(params, (long)transactionInput.getTransactionOutPointIndex(), new Sha256Hash(transactionInput.getTransactionOutPointHash().toByteArray())));
            if (transactionInput.hasSequence()) {
                input.setSequence(transactionInput.getSequence());
            }
            tx.addInput(input);
        }
    }

    private WalletTransaction connectTransactionOutputs(Protos.Transaction txProto, NetworkParameters params) {
        Transaction tx = this.txMap.get(txProto.getHash());
        WalletTransaction.Pool pool = WalletTransaction.Pool.valueOf(txProto.getPool().getNumber());
        for (int i = 0; i < tx.getOutputs().size(); ++i) {
            TransactionOutput output = tx.getOutputs().get(i);
            Protos.TransactionOutput transactionOutput = txProto.getTransactionOutput(i);
            if (!transactionOutput.hasSpentByTransactionHash()) continue;
            Transaction spendingTx = this.txMap.get(transactionOutput.getSpentByTransactionHash());
            int spendingIndex = transactionOutput.getSpentByTransactionIndex();
            output.markAsSpent(spendingTx.getInputs().get(spendingIndex));
        }
        if (txProto.hasConfidence()) {
            Protos.TransactionConfidence confidenceProto = txProto.getConfidence();
            TransactionConfidence confidence = tx.getConfidence();
            this.readConfidence(tx, confidenceProto, confidence);
        }
        return new WalletTransaction(pool, tx);
    }

    private void readConfidence(Transaction tx, Protos.TransactionConfidence confidenceProto, TransactionConfidence confidence) {
        if (!confidenceProto.hasType()) {
            log.warn("Unknown confidence type for tx {}", (Object)tx.getHashAsString());
            return;
        }
        TransactionConfidence.ConfidenceType confidenceType = TransactionConfidence.ConfidenceType.valueOf(confidenceProto.getType().getNumber());
        confidence.setConfidenceType(confidenceType);
        if (confidenceProto.hasAppearedAtHeight()) {
            if (confidence.getConfidenceType() != TransactionConfidence.ConfidenceType.BUILDING) {
                log.warn("Have appearedAtHeight but not BUILDING for tx {}", (Object)tx.getHashAsString());
                return;
            }
            confidence.setAppearedAtChainHeight(confidenceProto.getAppearedAtHeight());
        }
        if (confidenceProto.hasOverridingTransaction()) {
            if (confidence.getConfidenceType() != TransactionConfidence.ConfidenceType.OVERRIDDEN_BY_DOUBLE_SPEND) {
                log.warn("Have overridingTransaction but not OVERRIDDEN for tx {}", (Object)tx.getHashAsString());
                return;
            }
            Transaction overridingTransaction = this.txMap.get(confidenceProto.getOverridingTransaction());
            if (overridingTransaction == null) {
                log.warn("Have overridingTransaction that is not in wallet for tx {}", (Object)tx.getHashAsString());
                return;
            }
            confidence.setOverridingTransaction(overridingTransaction);
        }
    }
}

