//Credit to Paul Bourke (pbourke@swin.edu.au) for the original Fortran 77 Program :))
//Conversion to Visual Basic by EluZioN (EluZioN@casesladder.com)
//Conversion from VB to Delphi6 by Dr Steve Evans (steve@lociuk.com)
///////////////////////////////////////////////////////////////////////////////
//June 2002 Update by Dr Steve Evans (steve@lociuk.com): Heap memory allocation
//added to prevent stack overflow when MaxVertices and MaxTriangles are very large.
//Additional Updates in June 2002:
//Bug in InCircle function fixed. Radius r := Sqrt(rsqr).
//Check for duplicate points added when inserting new point.
//For speed, all points pre-sorted in x direction using quicksort algorithm and
//triangles flagged when no longer needed. The circumcircle centre and radius of
//the triangles are now stored to improve calculation time.
///////////////////////////////////////////////////////////////////////////////
//You can use this code however you like providing the above credits remain in tact

unit uDelaunay;

interface

uses Graphics, Types, uVrchol;

//Set these as applicable
Const MaxVertices   =   5000;
Const MaxTriangles  =  10000;
Const ExPtTolerance = 0.5;

//Points (Vertices)
Type dVertex = record
    x: Double;
    y: Double;
    PictVer: PVrchol;
end;

//Created Triangles, vv# are the vertex pointers
Type dTriangle = record
    vv0: LongInt;
    vv1: LongInt;
    vv2: LongInt;
    PreCalc: Integer;
    xc,yc,r: Double;
end;

type TDVertex = array[0..MaxVertices] of dVertex;
type PVertex = ^TDVertex;

type TDTriangle = array[0..MaxTriangles] of dTriangle;
type PTriangle = ^TDTriangle;

type TDComplete = array [0..MaxTriangles] of Boolean;
type PComplete = ^TDComplete;

type TDEdges = array[0..2,0..MaxTriangles * 3] of LongInt;
type PEdges = ^TDEdges;

type
  TShiftsArray = Class
    public
    width, height: integer;
    valYX: array of array of real;
    constructor Create(w, h, default: integer);
    destructor Destroy; override;
  end;

  TDelaunay = class
  private
    Vertex: PVertex; // vertices
    Triangle: PTriangle;
    HowMany: Integer;
    tPoints: Integer; //Variable for total number of points (vertices) +1
    triangulated: boolean;
    function InCircle(xp, yp, x1, y1, x2, y2, x3, y3: Double;
             var xc: Double; var yc: Double; var r: Double; j: Integer): Boolean;
    Function WhichSide(xp, yp, x1, y1, x2, y2: Double): Integer;
    Function Triangulate(nvert: Integer): Integer;
    procedure Mesh;
    procedure QuickSort(var A: PVertex; Low,High: Integer);
    procedure RenderTriangle(b1, b2, b3: PVrchol; SA: TShiftsArray);
  public
    constructor Create;
    destructor Destroy; override;
    procedure DrawLines (cL, cR: TCanvas);
    function CreateShiftsArray(x, y, w, h, default: integer): TShiftsArray;
    procedure AddPoint(x, y: Integer; Vrchol: PVrchol);
    procedure DeleteAll;
  end;

implementation

constructor TShiftsArray.Create(w, h, default: integer);
var
  i, j: integer;
begin
  height:= h;
  width:= w;
  SetLength(valYX, h);
  for i:= 0 to h-1 do begin
    SetLength(valYX[i], w);
    for j:= 0 to w-1 do
      valYX[i][j]:= default;
  end;
end;

destructor TShiftsArray.Destroy;
var
  i: integer;
begin
  for i:= 0 to height-1 do
    SetLength(valYX[i], 0);
  SetLength(valYX, 0);
end;

constructor TDelaunay.Create;
begin
  DeleteAll;
  //Allocate memory for arrays
  GetMem(Vertex, sizeof(Vertex^));
  GetMem(Triangle, sizeof(Triangle^));
end;

destructor TDelaunay.Destroy;
begin
  //Free memory for arrays
  FreeMem(Vertex, sizeof(Vertex^));
  FreeMem(Triangle, sizeof(Triangle^));
end;

procedure TDelaunay.DeleteAll;
begin
  //Initiate total points to 1, using base 0 causes problems in the functions
  tPoints := 1;
  HowMany:=0;
  triangulated:= false;
end;

function TDelaunay.InCircle(xp, yp, x1, y1, x2, y2, x3, y3: Double;
    var xc: Double; var yc: Double; var r: Double; j: Integer): Boolean;
//Return TRUE if the point (xp,yp) lies inside the circumcircle
//made up by points (x1,y1) (x2,y2) (x3,y3)
//The circumcircle centre is returned in (xc,yc) and the radius r
//NOTE: A point on the edge is inside the circumcircle
var
  eps: Double;
  m1: Double;
  m2: Double;
  mx1: Double;
  mx2: Double;
  my1: Double;
  my2: Double;
  dx: Double;
  dy: Double;
  rsqr: Double;
  drsqr: Double;
begin

  eps:= 0.000001;
  InCircle := False;

  //Check if xc,yc and r have already been calculated
  if  Triangle^[j].PreCalc=1 then
  begin
    xc := Triangle^[j].xc;
    yc := Triangle^[j].yc;
    r  := Triangle^[j].r;
    rsqr := r*r;
    dx := xp - xc;
    dy := yp - yc;
    drsqr := dx * dx + dy * dy;
  end else
  begin

    If (Abs(y1 - y2) < eps) And (Abs(y2 - y3) < eps) Then
    begin
     // ShowMessage('INCIRCUM - F - Points are coincident !!');
      Exit;
    end;

    If Abs(y2 - y1) < eps Then
    begin
      m2 := -(x3 - x2) / (y3 - y2);
      mx2 := (x2 + x3) / 2;
      my2 := (y2 + y3) / 2;
      xc := (x2 + x1) / 2;
      yc := m2 * (xc - mx2) + my2;
    end
    Else If Abs(y3 - y2) < eps Then
    begin
      m1 := -(x2 - x1) / (y2 - y1);
      mx1 := (x1 + x2) / 2;
      my1 := (y1 + y2) / 2;
      xc := (x3 + x2) / 2;
      yc := m1 * (xc - mx1) + my1;
    end
    Else
    begin
      m1 := -(x2 - x1) / (y2 - y1);
      m2 := -(x3 - x2) / (y3 - y2);
      mx1 := (x1 + x2) / 2;
      mx2 := (x2 + x3) / 2;
      my1 := (y1 + y2) / 2;
      my2 := (y2 + y3) / 2;
      if (m1-m2)<>0 then  //se
      begin
        xc := (m1 * mx1 - m2 * mx2 + my2 - my1) / (m1 - m2);
        yc := m1 * (xc - mx1) + my1;
      end else
      begin
        xc:= (x1+x2+x3)/3;
        yc:= (y1+y2+y3)/3;
      end;

    end;

    dx := x2 - xc;
    dy := y2 - yc;
    rsqr := dx * dx + dy * dy;
    r := Sqrt(rsqr);
    dx := xp - xc;
    dy := yp - yc;
    drsqr := dx * dx + dy * dy;

    //store the xc,yc and r for later use
    Triangle^[j].PreCalc:=1;
    Triangle^[j].xc:=xc;
    Triangle^[j].yc:=yc;
    Triangle^[j].r:=r;
  end;

  If drsqr <= rsqr Then InCircle := True;

end;



Function TDelaunay.WhichSide(xp, yp, x1, y1, x2, y2: Double): Integer;
//Determines which side of a line the point (xp,yp) lies.
//The line goes from (x1,y1) to (x2,y2)
//Returns -1 for a point to the left
//         0 for a point on the line
//        +1 for a point to the right
var
 equation: Double;
begin
  equation := ((yp - y1) * (x2 - x1)) - ((y2 - y1) * (xp - x1));

  If equation > 0 Then
     WhichSide := -1
  Else If equation = 0 Then
     WhichSide := 0
  Else
     WhichSide := 1;

End;



Function TDelaunay.Triangulate(nvert: Integer): Integer;
//Takes as input NVERT vertices in arrays Vertex()
//Returned is a list of NTRI triangular faces in the array
//Triangle(). These triangles are arranged in clockwise order.
var

  Complete: PComplete;
  Edges: PEdges;
  Nedge: LongInt;

  //For Super Triangle
  xmin: Double;
  xmax: Double;
  ymin: Double;
  ymax: Double;
  xmid: Double;
  ymid: Double;
  dx: Double;
  dy: Double;
  dmax: Double;

  //General Variables
  i : Integer;
  j : Integer;
  k : Integer;
  ntri : Integer;
  xc : Double;
  yc : Double;
  r : Double;
  inc : Boolean;
begin

  //Allocate memory
  GetMem(Complete, sizeof(Complete^));
  GetMem(Edges, sizeof(Edges^));


//Find the maximum and minimum Vertex bounds.
//This is to allow calculation of the bounding triangle
xmin := Vertex^[1].x;
ymin := Vertex^[1].y;
xmax := xmin;
ymax := ymin;
  For i := 2 To nvert do
  begin
  If Vertex^[i].x < xmin Then xmin := Vertex^[i].x;
  If Vertex^[i].x > xmax Then xmax := Vertex^[i].x;
  If Vertex^[i].y < ymin Then ymin := Vertex^[i].y;
  If Vertex^[i].y > ymax Then ymax := Vertex^[i].y;
  end;

dx := xmax - xmin;
dy := ymax - ymin;
If dx > dy Then
    dmax := dx
Else
    dmax := dy;

xmid := Trunc((xmax + xmin) / 2);
ymid := Trunc((ymax + ymin) / 2);

//Set up the supertriangle
//This is a triangle which encompasses all the sample points.
//The supertriangle coordinates are added to the end of the
//Vertex list. The supertriangle is the first triangle in
//the triangle list.

Vertex^[nvert + 1].x := (xmid - 2 * dmax);
Vertex^[nvert + 1].y := (ymid - dmax);
Vertex^[nvert + 2].x := xmid;
Vertex^[nvert + 2].y := (ymid + 2 * dmax);
Vertex^[nvert + 3].x := (xmid + 2 * dmax);
Vertex^[nvert + 3].y := (ymid - dmax);
Triangle^[1].vv0 := nvert + 1;
Triangle^[1].vv1 := nvert + 2;
Triangle^[1].vv2 := nvert + 3;
Triangle^[1].Precalc := 0;

Complete[1] := False;
ntri := 1;

//Include each point one at a time into the existing mesh
For i := 1 To nvert do
begin
  Nedge := 0;
    //Set up the edge buffer.
    //If the point (Vertex(i).x,Vertex(i).y) lies inside the circumcircle then the
    //three edges of that triangle are added to the edge buffer.
    j := 0;
      repeat
        j := j + 1;
        If Complete^[j] <> True Then
        begin
            inc := InCircle(Vertex^[i].x, Vertex^[i].y, Vertex^[Triangle^[j].vv0].x,
                            Vertex^[Triangle^[j].vv0].y, Vertex^[Triangle^[j].vv1].x,
                            Vertex^[Triangle^[j].vv1].y, Vertex^[Triangle^[j].vv2].x,
                            Vertex^[Triangle^[j].vv2].y, xc, yc, r,j);
            //Include this if points are sorted by X
            If (xc + r) < Vertex[i].x Then  //
               complete[j] := True          //
            Else                            //
            If inc Then
            begin
                Edges^[1, Nedge + 1] := Triangle^[j].vv0;
                Edges^[2, Nedge + 1] := Triangle^[j].vv1;
                Edges^[1, Nedge + 2] := Triangle^[j].vv1;
                Edges^[2, Nedge + 2] := Triangle^[j].vv2;
                Edges^[1, Nedge + 3] := Triangle^[j].vv2;
                Edges^[2, Nedge + 3] := Triangle^[j].vv0;
                Nedge := Nedge + 3;
                Triangle^[j].vv0 := Triangle^[ntri].vv0;
                Triangle^[j].vv1 := Triangle^[ntri].vv1;
                Triangle^[j].vv2 := Triangle^[ntri].vv2;
                Triangle^[j].PreCalc:=Triangle^[ntri].PreCalc;
                Triangle^[j].xc:=Triangle^[ntri].xc;
                Triangle^[j].yc:=Triangle^[ntri].yc;
                Triangle^[j].r:=Triangle^[ntri].r;
                Triangle^[ntri].PreCalc:=0;
                Complete^[j] := Complete^[ntri];
                j := j - 1;
                ntri := ntri - 1;
            End;
        End;
    until j>=ntri;

// Tag multiple edges
// Note: if all triangles are specified anticlockwise then all
// interior edges are opposite pointing in direction.
    For j := 1 To Nedge - 1 do
    begin
        If Not (Edges^[1, j] = 0) And Not (Edges^[2, j] = 0) Then
        begin
            For k := j + 1 To Nedge do
            begin
                If Not (Edges^[1, k] = 0) And Not (Edges^[2, k] = 0) Then
                begin
                    If Edges^[1, j] = Edges^[2, k] Then
                    begin
                        If Edges^[2, j] = Edges^[1, k] Then
                         begin
                            Edges^[1, j] := 0;
                            Edges^[2, j] := 0;
                            Edges^[1, k] := 0;
                            Edges^[2, k] := 0;
                         End;
                     End;
                End;
            end;
        End;
    end;

//  Form new triangles for the current point
//  Skipping over any tagged edges.
//  All edges are arranged in clockwise order.
    For j := 1 To Nedge do
    begin
            If Not (Edges^[1, j] = 0) And Not (Edges^[2, j] = 0) Then
            begin
                ntri := ntri + 1;
                Triangle^[ntri].vv0 := Edges^[1, j];
                Triangle^[ntri].vv1 := Edges^[2, j];
                Triangle^[ntri].vv2 := i;
                Triangle^[ntri].PreCalc:=0;
                Complete^[ntri] := False;
            End;
    end;
end;

//Remove triangles with supertriangle vertices
//These are triangles which have a Vertex number greater than NVERT
i:= 0;
repeat
  i := i + 1;
  If (Triangle^[i].vv0 > nvert) Or (Triangle^[i].vv1 > nvert) Or (Triangle^[i].vv2 > nvert) Then
  begin
    Triangle^[i].vv0 := Triangle^[ntri].vv0;
    Triangle^[i].vv1 := Triangle^[ntri].vv1;
    Triangle^[i].vv2 := Triangle^[ntri].vv2;
    i := i - 1;
    ntri := ntri - 1;
  End;
until i>=ntri;

Triangulate := ntri;

  //Free memory
  FreeMem(Complete, sizeof(Complete^));
  FreeMem(Edges, sizeof(Edges^));
End;

procedure TDelaunay.Mesh;
begin
  if triangulated then exit;
  QuickSort(Vertex,1,tPoints-1);
  If tPoints > 3 Then
  HowMany := Triangulate(tPoints-1); // Returns number of triangles created.
  triangulated:= true;
end;

procedure TDelaunay.DrawLines(cL, cR: TCanvas);
var
  i: Integer;
  body: array [0..3] of TPoint;
begin
  Mesh;
  //Draw the created triangles
  cL.Pen.Color:= clBlack;
  cR.Pen.Color:= clBlack;
  For i:= 1 To HowMany do
    with Triangle^[i] do begin
      body[0]:= Point(Vertex^[vv0].PictVer^.Lx, Vertex^[vv0].PictVer^.Ly);
      body[1]:= Point(Vertex^[vv1].PictVer^.Lx, Vertex^[vv1].PictVer^.Ly);
      body[2]:= Point(Vertex^[vv2].PictVer^.Lx, Vertex^[vv2].PictVer^.Ly);
      body[3]:= body[0];
      cL.Polyline(body);
      body[0]:= Point(Vertex^[vv0].PictVer^.Rx, Vertex^[vv0].PictVer^.Ry);
      body[1]:= Point(Vertex^[vv1].PictVer^.Rx, Vertex^[vv1].PictVer^.Ry);
      body[2]:= Point(Vertex^[vv2].PictVer^.Rx, Vertex^[vv2].PictVer^.Ry);
      body[3]:= body[0];
      cR.Polyline(body);
  end;
end;

procedure TDelaunay.RenderTriangle(b1, b2, b3: PVrchol; SA: TShiftsArray);
var
  temp: PVrchol;
  v1x, v1y, v2x, v2y,  {jednotkove vektory trojuholnika}
  det: integer;
  Xx, Xy, Yx, Yy,      {jednotkove vektory v pixelovej sustave}
  v1pos, v2pos,        {pozicia bodu v ramci trojuholnika}
  smernicaZac, smernicaKon: double;
  j, i, zac, kon, v1gradient, v2gradient, tmp: integer;
begin
{Akonahle su dva body zhodne, kasleme na to (nulovy obsah, delenie nulou v determinante)}
  if ((b1.x = b2.x) and (b1.y = b2.y)) or
     ((b2.x = b3.x) and (b2.y = b3.y)) or
     ((b3.x = b1.x) and (b3.y = b1.y))
  then exit;
{bubble sort}
  if b1.y > b2.y then begin temp:= b2; b2:= b1; b1:= temp end;
  if b2.y > b3.y then begin temp:= b3; b3:= b2; b2:= temp end;
  if b1.y > b2.y then begin temp:= b2; b2:= b1; b1:= temp end;
{Jednotkove vektory}
  v1x:= b2.x - b1.x;
  v1y:= b2.y - b1.y;
  v2x:= b3.x - b1.x;
  v2y:= b3.y - b1.y;
  det:= v1x*v2y - v2x*v1y;
  Xx:=  v2y/det;
  Xy:= -v1y/det;
  Yx:= -v2x/det;
  Yy:=  v1x/det;
{je trojuholik na priamke?}
  if v2y = 0 then exit;
{koeficienty rastu shiftu pri posune po vektoroch v1, v2}
  v1gradient:= b2.shift-b1.shift;
  v2gradient:= b3.shift-b1.shift;

  smernicaZac:= v2x/v2y;
{trojuholnik rozdelime vodorovne cez stredny vrchol}
  if v1y <> 0 then begin  {je v1 vodorovne => mozeme preskocit}
    smernicaKon:= v1x/v1y;
    for j:= 0 to v1y do begin
      zac:= round(j*smernicaZac);
      kon:= round(j*smernicaKon);
      if kon<zac then begin tmp:= kon; kon:= zac; zac:= tmp end;
      for i:= zac to kon do begin
        v1pos:= i*Xx + j*Yx;
        v2pos:= i*Xy + j*Yy;
        if (v1pos < 0) or (v2pos < 0) or ((v1pos + v2pos) > 1.00001)
        then continue;
//      SA.valYX[j+b1.y][i+b1.x]:= round(b1.shift + v1pos*v1gradient + v2pos*v2gradient);
        SA.valYX[j+b1.y][i+b1.x]:= b1.shift + v1pos*v1gradient + v2pos*v2gradient;
      end
    end;
  end;
{Druha polovica}
  if (b3.y-b2.y) <> 0 then begin  {je v3 vodorovne => mozeme preskocit}
    smernicaKon:= (b3.x-b2.x)/(b3.y-b2.y);
    for j:= v1y+1 to v2y do begin
      zac:= round(j*smernicaZac);
      kon:= round((j-v1y)*smernicaKon+v1x);
      if kon<zac then begin tmp:= kon; kon:= zac; zac:= tmp end;
      for i:= zac to kon do begin
        v1pos:= i*Xx + j*Yx;
        v2pos:= i*Xy + j*Yy;
        if (v1pos < 0) or (v2pos < 0) or ((v1pos + v2pos) > 1.00001)
        then continue;
//      SA.valYX[j+b1.y][i+b1.x]:= round(b1.shift + v1pos*v1gradient + v2pos*v2gradient);
        SA.valYX[j+b1.y][i+b1.x]:= b1.shift + v1pos*v1gradient + v2pos*v2gradient;
      end;
    end;
  end;
end;

function TDelaunay.CreateShiftsArray(x, y, w, h, default: integer): TShiftsArray;
var
  SA: TShiftsArray;
  i: integer;
begin
  Mesh;
  SA:= TShiftsArray.Create(w, h, default);
  for i:= 1 to HowMany do
    with Triangle^[i] do
      RenderTriangle(Vertex^[vv0].PictVer, Vertex^[vv1].PictVer,
        Vertex^[vv2].PictVer, SA);
  CreateShiftsArray:= SA;
end;

procedure TDelaunay.AddPoint(x, y: Integer; Vrchol: PVrchol);
{Vytvori novy bod na (x,y) a kerslit sa bude na prislusne obrazky cL a cR}
var
  i: Integer;
begin
  if tPoints > MaxVertices - 4 then exit;
  //Check for duplicate points
  for i:=1 to tPoints - 1 do
    If (Abs(x-Vertex^[i].x) < ExPtTolerance) and
       (Abs(y-Vertex^[i].y) < ExPtTolerance)
    then exit;

  Vertex^[tPoints].x := x;
  Vertex^[tPoints].y := y;
  Vertex^[tPoints].PictVer := Vrchol;
  //Increment the total number of points
  tPoints := tPoints + 1;
  triangulated:= false;
end;

procedure TDelaunay.QuickSort(var A: PVertex; Low,High: Integer);
//Sort all points by x
  procedure DoQuickSort(var A: PVertex; iLo, iHi: Integer);
  var
    Lo, Hi: Integer;
    Mid: Double;
    T: dVertex;
  begin
    Lo := iLo;
    Hi := iHi;
    Mid := A^[(Lo + Hi) div 2].x;
    repeat
      while A^[Lo].x < Mid do Inc(Lo);
      while A^[Hi].x > Mid do Dec(Hi);
      if Lo <= Hi then
      begin
        T := A^[Lo];
        A^[Lo] := A^[Hi];
        A^[Hi] := T;
        Inc(Lo);
        Dec(Hi);
      end;
    until Lo > Hi;
    if Hi > iLo then DoQuickSort(A, iLo, Hi);
    if Lo < iHi then DoQuickSort(A, Lo, iHi);
  end;
begin
  if Low < High then DoQuickSort(A, Low, High);
end;

end.
