from reservation import Reservation
import logging

logger = logging.getLogger(__name__)


class Library(object):
    def __init__(self, res_factory=Reservation):
        self._users = set()
        self._books = {}   #maps name to count
        self._reservations = [] #Reservations sorted by from
        self.res_factory=res_factory
        logger.info(F'Library created.')
                
    def add_user(self, name):
        if name in self._users:
            logger.info(F'User not created, user with name {name} already exists.')
            return False
        self._users.add(name)
        logger.info(F'User {name} created.')
        return True

    def add_book(self, name):
        self._books[name] = self._books.get(name, 0) + 1
        logger.info(F'Book {name} added. We have {self._books[name]} coppies of the book.')

    def reserve_book(self, user, book, date_from, date_to):
        book_count = self._books.get(book, 0)
        if user not in self._users:
            logger.info(F'We cannot reserve book {book} for {user} from {date_from} to {date_to}. '+
                  F'User does not exist.')
            return False
        if date_from > date_to:
            logger.warning(F'We cannot reserve book {book} for {user} from {date_from} to {date_to}. '+
                  F'Incorrect dates.')
            return False
        if book_count == 0:
            logger.info(F'We cannot reserve book {book} for {user} from {date_from} to {date_to}. '+
                  F'We do not have that book.')
            return False
        desired_reservation = self.res_factory(date_from, date_to, book, user)
        relevant_reservations = [res for res in self._reservations
                                 if desired_reservation.overlapping(res)] + [desired_reservation]
        #we check that if we add this reservation then for every reservation record that starts 
        #between date_from and date_to no more than book_count books are reserved.
        for from_ in [res.from_ for res in relevant_reservations]:
            if desired_reservation.includes(from_):
                if sum([rec.includes(from_) for rec in relevant_reservations]) > book_count:
                    logger.info(F'We cannot reserve book {book} for {user} from {date_from} '+
                          F'to {date_to}. We do not have enough books.')
                    return False
        self._reservations+=[desired_reservation]
        self._reservations.sort(key=lambda x:x.from_)
        logger.info(F'Reservation {desired_reservation._id} included.')
        return True

    def check_reservation(self, user, book, date):
        res = any([res.identify(date, book, user) == "OK" for res in self._reservations])
        logger.info(F'Reservation for {user} of {book} on {date} {"exists" if res else "does not exist"}.')
        return res

    def change_reservation(self, user, book, date, new_user):
        relevant_reservations = [res for res in self._reservations 
                                     if res.identify(date, book, user) == "OK"]
        if not relevant_reservations:
            logging.info(F'Reservation for {user} of {book} on {date} does not exist.')
            return False
        if new_user not in self._users:
            logging.info(F'Cannot change the reservation as {new_user} does not exist.')
            return False
            
        logging.info(F'Reservation for {user} of {book} on {date} changed to {new_user}.')  
        relevant_reservations[0].change_for(new_user)
        return True
