#include <iostream>
#include <fstream>
#include <random>
#include <vector>
#include <cassert>
#include <algorithm>
#include "dominating_set.h"


typedef struct {
    int seed;
    int vertices;
    int k;
    int answer1;
    int answer2;
} Config;


Config getConfig(int task) {
     switch(task) {
     case 1:
     	return {1, 8, 2, 1, 1};
     case 2:
     	return {1, 6, 2, 1, 0};
     case 3:
     	return {2, 8, 2, 0, 0};
     case 4:
     	return {0, 20, 5, 1, 1};
     case 5:
     	return {5, 20, 5, 0, 0};
     case 6:
     	return {5, 20, 6, 1, 0};
     case 7:
        return {51, 44, 11, 0, 0};
     case 8:
        return {52, 44, 11, 0, 0};
     case 9:
        return {52, 44, 12, 1, 0};
     case 10:
        return {53, 44, 12, 0, 0};
     case 11:
        return {53, 44, 13, 1, 0};
     case 12:
        return {7678, 44, 11, 1, 1};
     case 13:
        return {7696, 44, 11, 1, 1};
     };
     assert(false);
}


inline void write(const Cnf &cnf, const char* filename) {
    std::ofstream myfile;
    myfile.open(filename);
    myfile<<"p cnf "<<"100000"<<" "<<cnf.size()<<"\n";
    for(const Clause &c:cnf) {
        for(int literal:c) {
            myfile<<literal<<" ";
        }
#if PUT==1
            myfile<<" 0\n";
#else
            myfile<<"\n";
#endif
    }
    myfile.close();
}


int main() {
    const Config config = getConfig(VAL);

    const unsigned int n = config.vertices;
    
    std::mt19937 gen(config.seed); //set seed

    Graph g(n);
    while(1) {
        // We take a vertex with least number of elements in the list
        auto size_compare = [](auto x, auto y){ return x.size() < y.size(); };
        auto vertex_index = std::min_element(g.begin(), g.end(), size_compare) - g.begin();
        if (g[vertex_index].size() == 3) {
            break; // We have a cubic graph as all vertices already have degree three
        }

        std::vector<int> candidate_vertices;
        for(int i=0; i<n; i++) {
            if (i == vertex_index) continue;
            if (g[i].size() == 3) continue;
            bool multiedge = (std::find(g[i].begin(), g[i].end(), vertex_index) != g[i].end());
            if (multiedge) continue;
            candidate_vertices.push_back(i);
        }

        // If we are stuck we start from scratch
        if (candidate_vertices.size()==0) { 
            g=Graph(n);
            continue;
        }
        
        int other_vertex_index = candidate_vertices[gen()%candidate_vertices.size()];
        auto sorted_insert = [](auto &v, int b) {
            auto it = std::lower_bound(v.begin(), v.end(), b);
            v.insert(it, b);
        };
        sorted_insert(g[vertex_index], other_vertex_index);
        sorted_insert(g[other_vertex_index], vertex_index);
    }
    
    for(int i=0; i<n; i++) {
        std::cout<<"Vertex:"<<i<<", Adjacent:";
        for(int e: g[i]) {
            std::cout<<e<<",";
        }
        std::cout<<"\n";
    }
    
#if TASK==1
    std::cout<<"Searching for a set of size "<<config.k<<".\n";
    std::cout<<"Expected answer "<<config.answer1<<".\n";
    bool result = hasDominatingSetOfSizeK(g, config.k);
    std::cout << "Task 1 finished. Result:"<<result<<"\n";
#endif

#if TASK==2
    std::cout<<"Searching for a set of size "<<config.k<<".\n";
    std::cout<<"Expected answer "<<config.answer1<<".\n";
    Cnf res = hasDominatingSetOfSizeKToCNF(g, config.k);
    assert(res.size()>0);
    write(res, "lpi01.txt");
#endif

#if TASK==3
    std::cout<<"Searching for a set of size n/4.\n";
    std::cout<<"Expected answer "<<config.answer2<<".\n";
    Cnf res = hasDominatingSetOfSizeNOver4ToCNF(g);
    assert(res.size()>0);
    write(res, "lpi01.txt");
#endif


    return 0;
}
