<?php

/**
 * Object handling a postgres DB Table
 */
class pgTable{
  
  /**
   * @var object pgDB object storing an actual DB connection
   */
  private $db_conn;
  
  /**
   * @var array An associative array of table-row values
   */
  private $fields = array();
  
  /**
   * @var string Primary key
   */
  protected $key = 'id';
  
  /**
   * @var string Table name. It has to be changed in each individual Table Class
   */
  protected $table_name = "";
  
  /**
   * @var array errors in one array
   */
  private $error = array();
  
  /**
   * @var string last called SQL query
   */
  private $last_query;
  
  public function __construct($db_conn=null){
	//set a DB resource
	$this->db_conn = $db_conn!==NULL?$db_conn:DB_CONN;
	$this->setMetaData();
  }

  /**
   * @abstract Function make a single query on DB and set a error if any
   * @param string $sql A well-formatted SQL query
   * @return false on error
   */
  public function query($sql){
	$this->last_query = $sql;
	//echo $sql."\n";
	if(!$result = @pg_query($this->db_conn,$sql))
		$this->error(pg_last_error($this->db_conn),__FUNCTION__);
	return $result;
  }  

  /**
   * @abstract Function execute a single query using parametrized query and set a error if any
   * @param string $sql A well-formatted SQL query
   * @param array $params Array of params replacing $no. in query 
   * @return false on error
   * @example paramQuery("SELECT * FROM table WHERE name=$1",array('Joe'))
   */
  public function paramQuery($sql,$params){
	$this->last_query = $sql."; PARAMS:".implode(", ",$params);
	if(!$result = pg_query_params($this->db_conn,$sql,$params))
		$this->error(pg_last_error($this->db_conn),__FUNCTION__);
	return $result;
  }  

  /**
   * Function called in constructor sets the metadata of table and initialize the array of fields 
   */
  private function setMetaData() {      
    $sql = 'SELECT a.attnum, a.attname, t.typname, a.attlen, a.atttypmod, a.attnotnull, a.atthasdef FROM pg_class as c, pg_attribute a, pg_type t WHERE a.attnum > 0 and a.attrelid = c.oid and c.relname = '."'".$this->table_name."'".' and a.atttypid = t.oid order by a.attnum'; 
    $qid = $this->query($sql); 
    if (!is_resource($qid)) { 
        $this->error('MetaData(): Query Error - table does not exist',__FUNCTION__); 
        return null; 
    }
	while ($row =pg_fetch_assoc($qid)) {
		$this->fields[$row['attname']] = array(
			'value'=> null,
			'num'=>$row['attnum'], 
			'type'=>$row['typname'], 
			'length'=>$row['attlen'], 
			'not_null'=>$row['attnotnull'], 
			'has_default'=>$row['atthasdef'],
			'default_or_null'=>$row['atthasdef']?'default':'null',
			'slashes'=>true);
	}
  }
	  
  //funkcia nacita riadok do objektu ak najde prave jeden riadok splnajuci danu podmienku
  public function loadWhere($condition){
    $result = $this->query('SELECT * FROM "'.$this->table_name.'" WHERE '.$condition);
	if(!$result){
	  $this->error('Nepodarilo sa nacitat zaznam z DB.',__FUNCTION__);
	  return false; #ak nenacita ziaden riadok, nemame co robit a koncime
	}
 	if($row = pg_fetch_assoc($result))
       foreach($row as $key=>$value)
	     $this->fields[$key]['value'] = $value; 
	if(pg_num_rows($result)>1){
	  $this->error('SQL prikaz vratil viac riadkov.',__FUNCTION__); #ak nacitame viac riadkov, zaznacime chybu ale ostane nam v objekte prvy vrateny riadok
	  return false;
	}
	elseif(pg_num_rows($result)==0){
	  $this->error('SQL prikaz nevratil ziaden zaznam.',__FUNCTION__);
	  return false;
	}	
	else
	  return true;
  }

  //funkcia nacita riadok podla primarneho kluca
  public function load($key_value){
    return $this->loadWhere($this->key.'='.pg_escape_string($key_value));
  }
  
  public function reload(){
  	return $this->load($this->get($this->key));
  }
  
  //funkcia vrati hodnotu pola
  public function get($field_name){
    if(isset($this->fields[$field_name])){
 	  if($this->fields[$field_name]['type']=='bool')
 	  	return $this->getBool($field_name);
 	  else		
 	  	return $this->fields[$field_name]['value'];
    }
	else{
	  $this->error('Pole <b>'.$field_name.'</b> v tabulke '.$this->table_name.' neexistuje.',__FUNCTION__);
	  return false;
	}
  }

  /**
   * Funkcia nastavi hodnotu objektu v danom stlpci
   * @param string $field_name - nazov pola
   * @param mixed $new_value - nova hodnota
   * @param boolean $secure - ci escapovat vkladanu hodnotu. default: true;
   * @param boolean $slashes - ci pri vytvarani SQL prikazu uzvriet tuto hodnotu do uvodzoviek. default: true
   * 
   * Pouzitie: 
   *    1. Nastavit hodnotu od uzivatela
   * 	$table->set('comment',$_POST['comment']); 
   *    2. Nastavit cas vlozenia do DB 
   * 	$table->set('insert_date','NOW()',false,false);
   */
  public function set($field_name,$new_value,$secure=true,$slashes=true){
    if(isset($this->fields[$field_name])){
 	  if($new_value===null){//ak je nova hodnota null, tak to nemusime prehanat cez osetrenia, a vyhneme sa tak tomu, ze sa null prekonvertuje na prazdny retazec
 	  	$this->fields[$field_name]['value'] = null;
 	  	return true;
 	  }
 	  if($secure)
 	  	$this->fields[$field_name]['value'] = pg_escape_string($new_value);
 	  else
 	  	$this->fields[$field_name]['value'] = $new_value;
 	  $this->fields[$field_name]['slashes'] = $slashes;		
	  return true;
	}
	else{
	  $this->error('Pole <b>'.$field_name.'</b> v tabulke '.$this->table_name.' neexistuje.',__FUNCTION__);
	  return false;
	}
  }

  //funkcia ulozi hodnoty objektu do databazy - INSERT
  public function add(){
    $field_names = array();
	$values = array();
	foreach($this->fields as $field_name=>$field){
	  if($field_name != $this->key){
	    if($field['value']===null){ //ak je hodnota pola null, musime osetrit, ci tam aj moze byt a ci ju mame vkladat
		    if($field['not_null']=='t'){
		    	if($field['has_default']=='f'){
		    		$this->error('Nenastavena hodnota pola '.$field['name'].' vedie k chybe. Toto pole nemoze mat hodnotu null a nema nastavenu ani defaultnu hodnotu.',__FUNCTION__);
		    		return false;
		    	}
		    	else //ak ma default hodnotu, tak toto pole do sql prikazu davat nebudeme, postgres ju tam doplni automaticky
		    		continue;
		    }
		    else{ //ak mozem mat null
		    	if($field['default_or_null'] == 'default')
		    		continue; //ak nemam vyslovene naznacene, ze tam mam dat null, tak to do SQL prikazu nemusim davat vobec
		    	else{
				    $field_names[] = '"'.$field_name.'"';
				    $values[] = 'NULL';
		    	}	
		    }
	    }
	    else{ //ak je hodnota pola nenulova
		    $field_names[] = '"'.$field_name.'"';
		    $slash = $field['slashes']?"'":"";
		    $values[] = $slash.$field['value'].$slash;
	    }
	  }
	}
	$sql = 'INSERT INTO "'.$this->table_name.'" ('.implode(',',$field_names).') VALUES('.implode(',',$values).')';
	//echo $sql;
	if(!$this->query($sql)){
	  $this->error['add'] = 'Nepodarilo sa ulozit riadok do tabulky '.$this->table_name.'.';
	  return false;
	}
	$this->set($this->key,$this->oneFieldResult("select currval('".$this->table_name."_".$this->key."_seq')"));
	return true;
  }  

  //funkcia ulozi hodnoty objektu do databazy - UPDATE - kluc musi mat definovanu hodnotu
  public function save(){
    //ak nemame kluc - koncime, sorry kamo
    if(!$this->fields[$this->key]['value']){
	  $this->error('Nemozem spravit UPDATE, ked neviem, kde ho mam spravit. Nie je zadana polozka <b>'.$this->key.'</b>.',__FUNCTION__);
	  return false;
	}  
	$sql = 'UPDATE "'.$this->table_name.'" SET ';
	foreach($this->fields as $field_name=>$field){
	  if($field_name == $this->key)
	  	continue;
	  //ak nemame zadanu hodnotu a moze byt hodnota pola null, tak dame radsej null ako string (vyhenem sa chybe napriklad pri integeri)
	  if($field['not_null']=='f' && $field['value']==""){
	  	$savedValue = "null";
	  	$slash = "";
	  }
	  else{	
	  	$savedValue = $field['value'];
	  	$slash = $field['slashes']?"'":"";
	  }
	  $sql_update[] = '"'.$field_name.'"'."=".$slash.$savedValue.$slash;
	}  
	$sql .= implode(', ',$sql_update).' WHERE "'.$this->key.'"'."='".$this->fields[$this->key]['value']."'";
	//echo $sql;
	if(!$this->query($sql)){
	  $this->error('Nepodarilo sa ulozit riadok do tabulky '.$this->table_name.'.',__FUNCTION__);
	  return false;
	}
	return true;
  }
  
  //funkcia vymaze riadok alebo riadky z tabulky - DELETE - podla podmienky
  public function deleteWhere($condition){
    return $this->query('DELETE FROM '.$this->table_name.' WHERE '.$condition);
  }
  
  //funkcia vymaze riadok z tabulky, podla nastaveneho kluca
  public function delete(){
    //ak nemame kluc - koncime, ale... prezili sme vojnu, prezijeme aj blahobyt
    if(!$this->fields[$this->key]['value']){
	  $this->error('Nemozem spravit DELETE, ked neviem, kde ho mam spravit. Nie je zadana polozka <b>'.$this->key.'</b>.',__FUNCTION__);
	  return false;
	}  
	return $this->deleteWhere($this->key."='".$this->fields[$this->key]['value']."'");
  }
  
  /**
   * funkcia vrati pole objektov, podla query
   * pre vsetky riadky, ktore mi query vrati zoberiem ID a podla neho vytvorim novy objekt
   */ 
  public function getRows($sql){
    $object_name = get_class($this);
    //print $sql."<br/>";
	$result = $this->query($sql);
	if(!$result){
	  $this->error('Nepodarilo sa nacitat zaznam z DB. Pravdepodobne je chybny SQL dotaz.',__FUNCTION__);
      return false;
	}
	$bad_lines = 0;
	$rows = array();
	while($row = pg_fetch_assoc($result)){
	  if(isset($row[$this->key])){
	    $object = new $object_name();
	    if($object->load($row[$this->key]))
  		  $rows[] = $object;
  		else $bad_lines++;
	  }
	  else $bad_lines++;
	}
	if($bad_lines>0)
	  $this->error($bad_lines.' riadkov sa nepodarilo nacitat, nebol zadany kluc, alebo neexistuje riadok s tymto klucom.',__FUNCTION__);
	return $rows; 
  }
  
  public function getMyRows($where="1=1",$order=''){
  	if($order)
  		$orderstr = ' ORDER BY "'.implode('","',explode(",",$order)).'"';
  	return $this->getRows('SELECT * FROM "'.$this->table_name.'" WHERE ('.$where.')'.$orderstr);
  }
  
  //funkcia vracia vysledok dotazu SELECT COUNT()....
  public function countRows($sql,$field){
  	if($result = $this->query($sql))
  	  if($count = pg_fetch_assoc($result))
  	    return $count[$field];
  	return false;    
  }
  
  /**
   * funkcia nastavi hodnoty objektu podla udajov z asociativneho pola
   */
  public function loadFromArray($data){
  	if(!is_array($data)){
  		$this->error("Neplatny argument.. Ocakavam asociativne pole.",__FUNCTION__);
  		return false;
  	}
  	foreach($data as $key=>$value)
		$this->set($key,$value);
  	return true;			
  }

  /**
   * funkcia vrati asociativne pole hodnot objektu
   */
  public function getValues(){
	$tmp = array();
  	foreach($this->fields as $key=>$value)
  		$tmp[$key] = $value['value'];
  	return $tmp;			
  }  

  /**
   * nastavi polozku a ulozi riadok
   */  
  public function setSave($field,$value,$secure=true,$slashes=true){
  	 if(!$this->fields[$this->key]['value']){
  	 	$this->error("Nemam kluc pre tento zaznam, nemozem ulozit riadok.",__FUNCTION__);
  	 	return false;
  	 }
  	 elseif($this->set($field,$value,$secure,$slashes)){
 		$slash = $slashes?"'":"";
  	 	return $this->query('UPDATE "'.$this->table_name.'" SET "'.$field.'"'."=".$slash.$value.$slash." WHERE ".'"'.$this->key.'"'."='".$this->fields[$this->key]['value']."'");
  	 }
  }

  /**
   * Return PHP boolean value of boolean field in postgresDB
   */  
  public function getBool($field){
	$value = $this->fields[$field]['value'];
	if($value==='t')
		return true;
	elseif($value==='f')
		return false;
	else
		return null;		 
  }

  /**
   * Function which execute query and return first field of first row of result
   * @param string $query SQL query 
   */
  public function oneFieldResult($query){
  	if($result = $this->query($query))
  		return pg_fetch_result($result,0,0);
  	else
  		return false;	
  }

  
  /**
   * funkcia vracia naformatovany datum
   * funkcia dateFormat je v subore include/functions.php
   */
  public function getFormatedDate($field,$format){
  	return dateFormat($this->get($field),$format);
  } 
  
  /**
   * Riesi problem nullovej hodnoty pola.
   * Ak je hodnota null a je povolena nullova hodnota, neviem, ze ci mam vlozit do tabulky NULL alebo defaultnu hodnotu
   * Preto je vytvoreny priznak, ktory urci, co v takejto situacii robit.
   * V konstruktore je nastaveny na default, ale toto nastavenie sa da zmenit prave touto funckiou
   * @param string $field_name Name of fild in table
   * @param string $mode Can be 'default' or 'null' 
   */
  public function setDefaultOrNullParam($field_name,$mode){
    if(isset($this->fields[$field_name])){
		if($mode !== 'default' || $mode !== 'null'){
			$this->error("Mod musi byt 'default' alebo 'null' ",__FUNCTION__);
			return false;
		}
		if($this->fields[$field_name]['has_default'] !== 't' || $this->fields[$field_name]['not_null'] !== 't'){
			$this->error("Nemozes zmenit mod, ked pole {$field_name} nema defaultnu hodnotu, alebo nema povolenu NULL hodnotu.",__FUNCTION__);
			return false;
		}
	  	$this->fields[$field_name]['default_or_null'] = $mode;	
	 	return true;
	}
	else{
	  $this->error('Pole <b>'.$field_name.'</b> v tabulke '.$this->table_name.' neexistuje.',__FUNCTION__);
	  return false;
	}
  }

  /**
   * Funkcia, ktora zaznamena chybu
   */
  private function error($text,$function_name="none"){
  	global $error;
  	$error[get_class($this)][$function_name][] = array("text"=>$text,"params"=>$this->lastQuery());
  }
  
  
  public function debug(){
 	echo "chyba"; 	
  }
  /**
   * Funkcia vrati posledny SQL command vykonany cez $this->query()
   */
  public function lastQuery(){
  	return $this->last_query;
  }
  
  
  /**
   * parametre:
   * $textField="name",$idField="id",$whereOrSelect="1=1",$order="name",$firstEmpty=true,$selectParams="", $selected=null, $name=""
   * textField moze byt aj nejaky pattern, vyhodnocuje sa tak isto ako pattern v listingoch. Akurat ked tam nie je ziadna {} polozka, tak zistuje ci to nie je meno polozky
   * whereOrSelect - ak sa zacina na SELECT tak to berie ako jeden prikaz a pouzije getRows(). Inak to berie ako podmienku a pouzije getMyRows
   * ak nechame default hodnotu ako prazdny string, znamena to, ze mozeme default hodnotu vytiahnut z objektu
   * ak nechceme default hodnotu, tak ju nastavme ako null
   */
  public function getCombo($textField="name",$idField="id",$whereOrSelect="1=1",$order="name",$firstEmpty=true,$selectParams="", $selected=null, $name=""){
  	if(!$name)
  		$name = get_class($this)."Combo";
  	$out = "\n<select name='$name' $selectParams>";
  	if(preg_match("/^SELECT/i",$whereOrSelect)) //ak to je select, pouzijeme metodu getRows, inak getMyrows
  		$rows = $this->getRows($whereOrSelect);
  	else	
  		$rows = $this->getMyRows($whereOrSelect,$order);
  	if($selected === "")
  		$selected = $this->get($idField);
  	if($firstEmpty)
  		$out .= "\n\t<option></option>";
  	foreach($rows as $row){
  		$text = $row->get($textField)?$row->get($textField):$this->parseText($textField,$row);
  		$out .= "\n\t<option value='".$row->get($idField)."'";
  		if($selected==$row->get($idField))
  			$out .= " selected='selected'";
  		$out .=">".$text."</option>";
  	} 
  	$out .= "\n</select>";
  	return $out;
  }
  
  //ak dame vyraz do {} znamena to, ze chceme vratiti jeho hodnotu
  //ak dame vyraz do ## znamena to, ze to je vyraz, ktory treba vyhodnotit
  public function parseText($pattern,$row){
	//matchujeme {}
	preg_match_all("/\{([^}^{]+)\}/",$pattern,$matches);
	foreach($matches[1] as $key=>$columnName){
		$matches[0][$key] = '/'.$matches[0][$key].'/';  
	  	$matches[1][$key] = $row->get($columnName);
	}
	$pattern = preg_replace($matches[0],$matches[1],$pattern);

	//matchujeme ##
	preg_match_all("/#([^}^{]+)#/",$pattern,$matches);
	foreach($matches[1] as $key=>$term){
		$matches[0][$key] = '/'.str_replace(")","\)",str_replace("(","\(",str_replace("/","\/",$matches[0][$key]))).'/';  
	  	eval('$matches[1][$key] = '.$term.';');
	}
	return preg_replace($matches[0],$matches[1],$pattern);
  }
  
}

?>